# $Id: xively.py 1130 2014-11-24 13:13:36Z mwall $
# Copyright 2013 Matthew Wall
"""
Upload data to Xively (aka COSM, aka Pachube)
  https://xively.com/

[StdRESTful]
    [[Xively]]
        token = TOKEN
        feed = FEED_ID
"""

# FIXME: support all schema fields with per-field  names, formats, and units

import Queue
import sys
import syslog
import time
import urllib
import urllib2

import weewx
import weewx.restx
from weeutil.weeutil import to_bool, accumulateLeaves

try:
    import cjson as json
    # XXX: maintain compatibility w/ json module
    setattr(json, 'dumps', json.encode)
    setattr(json, 'loads', json.decode)
except Exception, e:
    try:
        import simplejson as json
    except Exception, e:
        import json

VERSION = "0.3rc2"

if weewx.__version__ < "3":
    raise weewx.UnsupportedFeature("weewx 3 is required, found %s" %
                                   weewx.__version__)

def logmsg(level, msg):
    syslog.syslog(level, 'restx: Xively: %s' % msg)

def logdbg(msg):
    logmsg(syslog.LOG_DEBUG, msg)

def loginf(msg):
    logmsg(syslog.LOG_INFO, msg)

def logerr(msg):
    logmsg(syslog.LOG_ERR, msg)

def _compat(d, old_label, new_label):
    if d.has_key(old_label) and not d.has_key(new_label):
        d.setdefault(new_label, d[old_label])
        d.pop(old_label)

class Xively(weewx.restx.StdRESTbase):
    def __init__(self, engine, config_dict):
        """This service recognizes standard restful options plus the following:

        Required parameters:

        token: unique token

        feed: the feed name

        Optional parameters:

        prefix: if specified it will be prepended to data names
        Default is None
        """
        super(Xively, self).__init__(engine, config_dict)
        loginf('service version is %s' % VERSION)
        try:
            site_dict = config_dict['StdRESTful']['Xively']
            site_dict = accumulateLeaves(site_dict, max_level=1)
            site_dict['feed']
            site_dict['token']
        except KeyError, e:
            logerr("Data will not be posted: Missing option %s" % e)
            return
        site_dict['manager_dict'] = weewx.manager.get_manager_dict(
            config_dict['DataBindings'], config_dict['Databases'], 'wx_binding')

        # for backward compatibility: 'station' is now 'prefix'
        _compat(site_dict, 'station', 'prefix')

        self.archive_queue = Queue.Queue()
        self.archive_thread = XivelyThread(self.archive_queue, **site_dict)
        self.archive_thread.start()
        self.bind(weewx.NEW_ARCHIVE_RECORD, self.new_archive_record)
        loginf("Data will be uploaded to feed %s" % site_dict['feed'])

    def new_archive_record(self, event):
        self.archive_queue.put(event.record)

class XivelyThread(weewx.restx.RESTThread):

    _SERVER_URL = 'http://api.xively.com/v2/feeds'
    _FORMATS = {'barometer'   : 'barometer %.3f',        # inHg
                'outTemp'     : 'temperature_out %.1f',  # F
                'outHumidity' : 'humidity_out %03.0f',   # %
                'inTemp'      : 'temperature_in %.1f',   # F
                'inHumidity'  : 'humidity_in %03.0f',    # %
                'windSpeed'   : 'windSpeed %.2f',        # mph
                'windDir'     : 'windDir %03.0f',        # compass degree
                'windGust'    : 'windGust %.2f',         # mph
                'dewpoint'    : 'dewpoint %.1f',         # F
                'rain24'      : 'rain24 %.2f',           # in
                'hourRain'    : 'hourRain %.2f',         # in
                'dayRain'     : 'dayRain %.2f',          # in
                'radiation'   : 'radiation %.2f',        # W/m^2
                'UV'          : 'UV %.2f'}               # number

    def __init__(self, queue, feed, token, manager_dict,
                 prefix=None, server_url=_SERVER_URL, skip_upload=False,
                 post_interval=None, max_backlog=sys.maxint, stale=None,
                 log_success=True, log_failure=True,
                 timeout=60, max_tries=3, retry_wait=5):
        super(XivelyThread, self).__init__(queue,
                                           protocol_name='Xively',
                                           manager_dict=manager_dict,
                                           post_interval=post_interval,
                                           max_backlog=max_backlog,
                                           stale=stale,
                                           log_success=log_success,
                                           log_failure=log_failure,
                                           max_tries=max_tries,
                                           timeout=timeout,
                                           retry_wait=retry_wait)
        self.feed = feed
        self.token = token
        self.prefix = prefix
        self.server_url = server_url
        self.skip_upload = to_bool(skip_upload)

    def process_record(self, record, dbm):
        r = self.get_record(record, dbm)
        url = self.get_url()
        data = self.get_data(r)
        if self.skip_upload:
            logdbg("skipping upload")
            return
        req = urllib2.Request(url, data)
        req.add_header("User-Agent", "weewx/%s" % weewx.__version__)
        req.add_header("X-PachubeApiKey", self.token)
        req.get_method = lambda: 'PUT'
        self.post_with_retries(req)

    def check_response(self, response):
        txt = response.read()
        if txt != '':
            raise weewx.restx.FailedPost(txt)

    def get_url(self):
        url = '%s/%s' % (self.server_url, self.feed)
        logdbg('url: %s' % url)
        return url
        
    def get_data(self, record):
        prefix = urllib.quote_plus(self.prefix) \
            if self.prefix is not None else None
        tstr = time.strftime('%Y-%m-%dT%H:%M:%SZ',
                             time.gmtime(record['dateTime']))
        streams = {}
        for k in self._FORMATS:
            v = record.get(k)
            if v is not None:
                dskey = '%s_%s' % (prefix, k) if prefix is not None else k
                if not dskey in streams:
                    streams[dskey] = {'id':dskey, 'datapoints':[]}
                dp = {'at':tstr, 'value':v}
                streams[dskey]['datapoints'].append(dp)
        if len(streams.keys()) == 0:
            return None
        data = {
            'version':'1.0.0',
            'datastreams':[]
            }
        for k in streams.keys():
            data['datastreams'].append(streams[k])
        data = json.dumps(data)
        logdbg('data: %s' % data)
        return data
