# $Id: wcloud.py 1080 2014-10-22 13:43:08Z mwall $
# Copyright 2014 Matthew Wall

"""
This is a weewx extension that uploads data to WeatherCloud.

http://weather.weathercloud.com

Based on weathercloud API documentation v0.5 as of 15oct2014.

The preferred upload frequency (post_interval) is one record every 10 minutes.

Minimal Configuration:

[StdRESTful]
    [[WeatherCloud]]
        id = WEATHERCLOUD_ID
        key = WEATHERCLOUD_KEY
"""

import Queue
import sys
import syslog
import time
import urllib
import urllib2

import weewx
import weewx.restx
import weewx.units
import weewx.wxformulas
from weeutil.weeutil import to_bool

VERSION = "0.5"

def logmsg(level, msg):
    syslog.syslog(level, 'restx: WeatherCloud: %s' % msg)

def logdbg(msg):
    logmsg(syslog.LOG_DEBUG, msg)

def loginf(msg):
    logmsg(syslog.LOG_INFO, msg)

def logerr(msg):
    logmsg(syslog.LOG_ERR, msg)

# weewx uses a status of 1 to indicate failure, wcloud uses 0
def _invert(x):
    if x == 0:
        return 1
    return 0

# utility to convert to METRICWX windspeed
def _convert_windspeed(v, from_unit_system):
    if from_unit_system is None:
        return None
    if from_unit_system != weewx.METRICWX:
        (from_unit, _) = weewx.units.getStandardUnitType(from_unit_system,
                                                         'windSpeed')
        from_t = (v, from_unit, 'group_speed')
        to_t = weewx.units.convert(from_t, 'meter_per_second')
        v = to_t.value
    return v

# FIXME: this formula is suspect
def _calc_thw(heatindex_C, windspeed_mps):
    if heatindex_C is None or windspeed_mps is None:
        return None
    windspeed_mph = 2.25 * windspeed_mps
    heatindex_F = 32 + heatindex_C * 9 / 5
    thw_F = heatindex_F - (1.072 * windspeed_mph)
    thw_C = (thw_F - 32) * 5 / 9
    return thw_C

def _get_windavg(archive, ts, interval=600):
    sts = ts - interval
    val = archive.getSql("SELECT AVG(windSpeed) FROM archive "
                         "WHERE dateTime>? AND dateTime<=?",
                         (sts, ts))
    if val is None:
        return None
    return val[0]

# weathercloud wants "10-min maximum gust of wind".  some hardware reports
# a wind gust, others do not, so try to deal with both.
def _get_windhi(archive, ts, interval=600):
    sts = ts - interval
    val = archive.getSql("""SELECT
 MAX(CASE WHEN windSpeed >= windGust THEN windSpeed ELSE windGust END)
 FROM archive
 WHERE dateTime>? AND dateTime<=?""",
                         (sts, ts))
    if val is None:
        return None
    return val[0]

def _get_winddiravg(archive, ts, interval=600):
    sts = ts - interval
    val = archive.getSql("SELECT AVG(windDir) FROM archive "
                         "WHERE dateTime>? AND dateTime<=?",
                         (sts, ts))
    if val is None:
        return None
    return val[0]

class WeatherCloud(weewx.restx.StdRESTbase):
    """Upload using the WeatherCloud protocol."""

    def __init__(self, engine, config_dict):
        """Initialize for upload to WeatherCloud.  This service supports the
        standard restful options plus the following:

        Required parameters:

        id: WeatherCloud identifier

        key: WeatherCloud key
        """
        super(WeatherCloud, self).__init__(engine, config_dict)
        loginf("service version is %s" % VERSION)
        try:
            site_dict = weewx.restx.get_dict(config_dict, 'WeatherCloud')
            site_dict['id']
            site_dict['key']
        except KeyError, e:
            logerr("Data will not be posted: Missing option %s" % e)
            return
        site_dict.setdefault('database_dict', config_dict['Databases'][config_dict['StdArchive']['archive_database']])

        self.archive_queue = Queue.Queue()
        self.archive_thread = WeatherCloudThread(self.archive_queue, **site_dict)
        self.archive_thread.start()
        self.bind(weewx.NEW_ARCHIVE_RECORD, self.new_archive_record)
        loginf("Data will be uploaded for id=%s" % site_dict['id'])

    def new_archive_record(self, event):
        self.archive_queue.put(event.record)

class WeatherCloudThread(weewx.restx.RESTThread):

    _SERVER_URL = 'http://api.weathercloud.net/v01/set'

    # this data map supports the default database schema
    # FIXME: design a config option to override this map
    #             wcloud_name   weewx_name      format  multiplier
    _DATA_MAP = {'temp':       ('outTemp',      '%.0f', 10.0), # C * 10
                 'hum':        ('outHumidity',  '%.0f', 1.0),  # percent
                 'wdir':       ('windDir',      '%.0f', 1.0),  # degree
                 'wspd':       ('windSpeed',    '%.0f', 10.0), # m/s * 10
                 'bar':        ('barometer',    '%.0f', 10.0), # hPa * 10
                 'rain':       ('dayRain',      '%.0f', 10.0), # mm * 10
                 'rainrate':   ('rainRate',     '%.0f', 10.0), # mm/hr * 10
                 'tempin':     ('inTemp',       '%.0f', 10.0), # C * 10
                 'humin':      ('inHumidity',   '%.0f', 1.0),  # percent
                 'uvi':        ('UV',           '%.0f', 10.0), # index * 10
                 'solarrad':   ('radiation',    '%.0f', 10.0), # W/m^2 * 10
                 'et':         ('EV',           '%.0f', 10.0), # mm * 10
                 'chill':      ('windchill',    '%.0f', 10.0), # C * 10
                 'heat':       ('heatindex',    '%.0f', 10.0), # C * 10
                 'dew':        ('dewpoint',     '%.0f', 10.0), # C * 10
                 'battery':    ('consBatteryVoltage', '%.0f', 100.0), # V * 100
                 'temp01':     ('extraTemp1',   '%.0f', 10.0), # C * 10
                 'temp02':     ('extraTemp2',   '%.0f', 10.0), # C * 10
                 'temp03':     ('extraTemp3',   '%.0f', 10.0), # C * 10
                 'temp04':     ('leafTemp1',    '%.0f', 10.0), # C * 10
                 'temp05':     ('leafTemp2',    '%.0f', 10.0), # C * 10
                 'temp06':     ('soilTemp1',    '%.0f', 10.0), # C * 10
                 'temp07':     ('soilTemp2',    '%.0f', 10.0), # C * 10
                 'temp08':     ('soilTemp3',    '%.0f', 10.0), # C * 10
                 'temp09':     ('soilTemp4',    '%.0f', 10.0), # C * 10
                 'temp10':     ('heatingTemp4', '%.0f', 10.0), # C * 10
                 'leafwet01':  ('leafWet1',     '%.0f', 1.0),  # [0,15]
                 'leafwet02':  ('leafWet2',     '%.0f', 1.0),  # [0,15]
                 'hum01':      ('extraHumid1',  '%.0f', 1.0),  # percent
                 'hum02':      ('extraHumid2',  '%.0f', 1.0),  # percent
                 'soilmoist01': ('soilMoist1',  '%.0f', 1.0),  # Cb [0,200]
                 'soilmoist02': ('soilMoist2',  '%.0f', 1.0),  # Cb [0,200]
                 'soilmoist03': ('soilMoist3',  '%.0f', 1.0),  # Cb [0,200]
                 'soilmoist04': ('soilMoist4',  '%.0f', 1.0),  # Cb [0,200]

                 # these are calculated by this extension
#                 'thw':        ('thw',          '%.0f', 10.0), # C * 10
                 'wspdhi':     ('windhi',       '%.0f', 10.0), # m/s * 10
                 'wspdavg':    ('windavg',      '%.0f', 10.0), # m/s * 10
                 'wdiravg':    ('winddiravg',   '%.0f', 1.0),  # degree
                 'heatin':     ('inheatindex',  '%.0f', 10.0), # C * 10
                 'dewin':      ('indewpoint',   '%.0f', 10.0), # C * 10
                 'battery01':  ('bat01',        '%.0f', 1.0),  # 0 or 1
                 'battery02':  ('bat02',        '%.0f', 1.0),  # 0 or 1
                 'battery03':  ('bat03',        '%.0f', 1.0),  # 0 or 1
                 'battery04':  ('bat04',        '%.0f', 1.0),  # 0 or 1
                 'battery05':  ('bat05',        '%.0f', 1.0),  # 0 or 1

                 # these are in the wcloud api but are not yet implemented
#                 'tempagroXX':   ('??',       '%.0f', 10.0), # C * 10
#                 'wspdXX':       ('??',       '%.0f', 10.0), # m/s * 10
#                 'wspdavgXX':    ('??',       '%.0f', 10.0), # m/s * 10
#                 'wspdhiXX':     ('??',       '%.0f', 10.0), # m/s * 10
#                 'wdirXX':       ('??',       '%.0f', 1.0), # degree
#                 'wdiravgXX':    ('??',       '%.0f', 1.0), # degree
#                 'bartrend':     ('??',       '%.0f', 1.0), # -60,-20,0,20,60
#                 'forecast':     ('??',       '%.0f', 1.0),
#                 'forecasticon': ('??',       '%.0f', 1.0),
                 }

    def __init__(self, queue, id, key, database_dict,
                 server_url=_SERVER_URL, skip_upload=False,
                 post_interval=600, max_backlog=sys.maxint, stale=None,
                 log_success=True, log_failure=True,
                 timeout=60, max_tries=3, retry_wait=5):
        super(WeatherCloudThread, self).__init__(queue,
                                               protocol_name='WeatherCloud',
                                               database_dict=database_dict,
                                               post_interval=post_interval,
                                               max_backlog=max_backlog,
                                               stale=stale,
                                               log_success=log_success,
                                               log_failure=log_failure,
                                               max_tries=max_tries,
                                               timeout=timeout,
                                               retry_wait=retry_wait)
        self.id = id
        self.key = key
        self.server_url = server_url
        self.skip_upload = to_bool(skip_upload)
        self.archive_units = None

    def process_record(self, record, archive):
        r = self.get_record(record, archive)
        url = self.get_url(r)
        if self.skip_upload:
            logdbg("skipping upload")
            return
        req = urllib2.Request(url)
        req.add_header("User-Agent", "weewx/%s" % weewx.__version__)
        self.post_with_retries(req)

    # calculate derived quantities and other values needed by wcloud
    def get_record(self, record, archive):
        if self.archive_units is None:
            u = archive.getSql("SELECT usUnits FROM archive LIMIT 1")
            if u is not None:
                self.archive_units = u[0]
        rec = super(WeatherCloudThread, self).get_record(record, archive)

        # put everything into units required by weathercloud
        rec = weewx.units.to_METRICWX(rec)

        # calculate additional quantities
        rec['windavg'] = _get_windavg(archive, record['dateTime'])
        rec['windhi'] = _get_windhi(archive, record['dateTime'])
        rec['winddiravg'] = _get_winddiravg(archive, record['dateTime'])

        # these observations are non-standard, so do unit conversions directly
        rec['windavg'] = _convert_windspeed(rec['windavg'], self.archive_units)
        rec['windhi'] = _convert_windspeed(rec['windhi'], self.archive_units)

        if 'inTemp' in rec and 'inHumidity' in rec:
            rec['inheatindex'] = weewx.wxformulas.heatindexC(
                rec['inTemp'], rec['inHumidity'])
            rec['indewpoint'] = weewx.wxformulas.dewpointC(
                rec['inTemp'], rec['inHumidity'])
#        if 'heatindex' in rec and 'windSpeed' in rec:
#            rec['thw'] = _calc_thw(rec['heatindex'], rec['windSpeed'])
        if record.has_key('txBatteryStatus'):
            rec['bat01'] = _invert(record['txBatteryStatus'])
        if record.has_key('windBatteryStatus'):
            rec['bat02'] = _invert(record['windBatteryStatus'])
        if record.has_key('rainBatteryStatus'):
            rec['bat03'] = _invert(record['rainBatteryStatus'])
        if record.has_key('outTempBatteryStatus'):
            rec['bat04'] = _invert(record['outTempBatteryStatus'])
        if record.has_key('inTempBatteryStatus'):
            rec['bat05'] = _invert(record['inTempBatteryStatus'])
        return rec

    def get_url(self, record):
        # put data into expected structure and format
        values = {}
        values['ver'] = str(weewx.__version__)
        values['type'] = 251 # identifier assigned to weewx by weathercloud
        values['wid'] = self.id
        values['key'] = self.key
        time_tt = time.gmtime(record['dateTime'])
        values['time'] = time.strftime("%H%M", time_tt) # assumes leading zeros
        for key in self._DATA_MAP:
            rkey = self._DATA_MAP[key][0]
            if record.has_key(rkey) and record[rkey] is not None:
                v = record[rkey] * self._DATA_MAP[key][2]
                values[key] = self._DATA_MAP[key][1] % v
        url = self.server_url + '?' + urllib.urlencode(values)
        logdbg('url: %s' % url)
        return url
