# $Id: twitter.py 1052 2014-10-14 20:11:46Z mwall $
# Copyright 2014 Matthew Wall
"""
Tweet weather data

You must first obtain app and oauth credentials in order to tweet.  See the
twitter developer documentation to obtain these:

  https://dev.twitter.com/oauth/overview

Specifically, you will need four things:  app_key, app_key_secret, oauth_token,
and oauth_token_secret.

Installation:

0) install twython
   pip install twython

1) put this file in the weewx user directory
   cp twitter.py /home/weewx/bin/user

2) add the following configuration stanza to weewx.conf

[StdRESTful]
    [[Twitter]]
        station = STATION_IDENTIFIER
        app_key = APP_KEY
        app_key_secret = APP_KEY_SECRET
        oauth_token = OAUTH_TOKEN
        oauth_token_secret = OAUTH_TOKEN_SECRET

[Engines]
    [[WxEngine]]
        restful_services = ... , user.twitter.Twitter

3) restart weewx
    sudo /etc/init.d/weewx stop
    sudo /etc/init.d/weewx start

By default, the units are those specified by the unit system in the
StdConvert section of weewx.conf.  To specify a different unit system,
use the unit_system option:

[StdRESTful]
    [[Twitter]]
        unit_system = METRICWX

Possible values include US, METRIC, or METRICWX.
"""

import Queue
import sys
import syslog
import time

import weewx
import weewx.restx
import weewx.units

from twython import Twython, TwythonError, TwythonAuthError, TwythonRateLimitError

VERSION = "0.4"

def logmsg(level, msg):
    syslog.syslog(level, 'restx: Twitter: %s' % msg)

def logdbg(msg):
    logmsg(syslog.LOG_DEBUG, msg)

def loginf(msg):
    logmsg(syslog.LOG_INFO, msg)

def logerr(msg):
    logmsg(syslog.LOG_ERR, msg)

def _format(label, fmt, datum):
    s = fmt % datum if datum is not None else "None"
    return "%s: %s" % (label, s)

class Twitter(weewx.restx.StdRESTbase):
    def __init__(self, engine, config_dict):
        """Initialize for posting to twitter.  This service supports the
        standard restful options plus the following:

        Required parameters:

        station: a short name to identify the weather station

        twitter authentication credentials:
        app_key
        app_key_secret
        oauth_token
        oauth_token_secret

        Optional parameters:

        unit_system: one of US, METRIC, or METRICWX
        Default is None
        """
        super(Twitter, self).__init__(engine, config_dict)
        loginf('service version is %s' % VERSION)
        try:
            site_dict = weewx.restx.get_dict(config_dict, 'Twitter')
            site_dict['station']
            site_dict['app_key']
            site_dict['app_key_secret']
            site_dict['oauth_token']
            site_dict['oauth_token_secret']
        except KeyError, e:
            logerr("Data will not be posted: Missing option %s" % e)
            return

        # if a unit system was specified, get the weewx constant for it
        usn = site_dict.get('unit_system', None)
        if usn is not None:
            site_dict['unit_system'] = weewx.units.unit_constants[usn]
            loginf('units will be converted to %s' % usn)

        # we can bind to archive or loop events, default to archive
        self.binding = site_dict.get('binding', 'archive')

        self.loop_queue = Queue.Queue()
        self.loop_thread = TwitterThread(self.loop_queue, **site_dict)
        self.loop_thread.start()
        if self.binding == 'loop':
            self.bind(weewx.NEW_LOOP_PACKET, self.handle_new_loop)
        else:
            self.bind(weewx.NEW_ARCHIVE_RECORD, self.handle_new_archive)
        loginf('binding is %s' % self.binding)
        loginf("Data will be tweeted for %s" % site_dict['station'])

    def handle_new_loop(self, event):
        self.loop_queue.put(event.packet)

    def handle_new_archive(self, event):
        self.loop_queue.put(event.record)

class TwitterThread(weewx.restx.RESTThread):
    _FORMATS = {
        "barometer":   ["P",  "%.3f"],
        "inTemp":      ["iT", "%.2f"],
        "outTemp":     ["oT", "%.2f"],
        "inHumidity":  ["iH", "%.2f"],
        "outHumidity": ["oH", "%.2f"],
        "windSpeed":   ["Ws", "%.1f"],
        "windGust":    ["Wg", "%.1f"],
        "windDir":     ["Wd", "%.0f"],
        "rain":        ["R",  "%.3f"],
        }

    def __init__(self, queue, 
                 station,
                 app_key, app_key_secret, oauth_token, oauth_token_secret,
                 unit_system=None, 
                 log_success=True, log_failure=True,
                 post_interval=None, max_backlog=sys.maxint, stale=None,
                 timeout=60, max_tries=3, retry_wait=5):
        super(TwitterThread, self).__init__(queue,
                                            protocol_name='Twitter',
                                            database_dict=None,
                                            post_interval=post_interval,
                                            max_backlog=max_backlog,
                                            stale=stale,
                                            log_success=log_success,
                                            log_failure=log_failure,
                                            max_tries=max_tries,
                                            timeout=timeout,
                                            retry_wait=retry_wait)
        self.unit_system = unit_system
        self.station = station
        self.app_key = app_key
        self.app_key_secret = app_key_secret
        self.oauth_token = oauth_token
        self.oauth_token_secret = oauth_token_secret

    def process_record(self, record, dummy_archive):
        if self.unit_system is not None:
            record = weewx.units.to_std_system(record, self.unit_system)
        s = []
        s.append("%s: ts: %s" % (self.station, record['dateTime']))
        for obs in self._FORMATS:
            if obs in record:
                s.append(_format(self._FORMATS[obs][0], self._FORMATS[obs][1],
                                 record[obs]))
        wstr = '; '.join(s)
        ntries = 0
        while ntries < self.max_tries:
            ntries += 1
            try:
                twitter = Twython(self.app_key, self.app_key_secret,
                                  self.oauth_token, self.oauth_token_secret)
                twitter.update_status(status=wstr)
                return
            except (TwythonError, TwythonAuthError, TwythonRateLimitError), e:
                logerr("Failed attempt %d of %d to tweet: %s" %
                       (ntries, self.max_tries, e))
                logdbg("Waiting %d seconds before retry" % self.retry_wait)
                time.sleep(self.retry_wait)
        else:
            raise weewx.restx.FailedPost("Max retries (%d) exceeded" %
                                         self.max_tries)
