# $Id$
# Copyright 2013 Matthew Wall
"""weewx module that records process information."""

from __future__ import with_statement
import os
import platform
import re
import syslog
import time
from subprocess import Popen, PIPE

import weewx
import weeutil.weeutil
from weewx.wxengine import StdService

def logmsg(level, msg):
    syslog.syslog(level, 'pmon: %s' % msg)

def logdbg(msg):
    logmsg(syslog.LOG_DEBUG, msg)

def loginf(msg):
    logmsg(syslog.LOG_INFO, msg)

def logerr(msg):
    logmsg(syslog.LOG_ERR, msg)

defaultSchema = [
    ('dateTime', 'INTEGER NOT NULL PRIMARY KEY'),
    ('usUnits', 'INTEGER'),
    ('mem_vsz','INTEGER'),
    ('mem_rss','INTEGER'),
    ]

class ProcessMonitor(StdService):

    def __init__(self, engine, config_dict):
        super(ProcessMonitor, self).__init__(engine, config_dict)

        d = config_dict.get('ProcessMonitor', {})
        self.process = d['process']
        self.max_age = weeutil.weeutil.to_int(d.get('max_age', 2592000))

        # get the database parameters we need to function
        self.database = d['database']
        self.table = d.get('table', 'archive')
        schema = defaultSchema

        # configure the database
        self.archive = weewx.archive.Archive.open_with_create(config_dict['Databases'][self.database], schema, self.table)

        # be sure database matches the schema we have
        dbcol = self.archive.connection.columnsOf(self.table)
        memcol = [x[0] for x in schema]
        if dbcol != memcol:
            raise Exception('schema mismatch: %s != %s' % (dbcol, memcol))

        self.bind(weewx.NEW_ARCHIVE_RECORD, self.newArchiveRecordCallback)

    def shutDown(self):
        pass

    def newArchiveRecordCallback(self, event):
        """save data to database then prune old records as needed"""
        now = int(time.time())
        delta = now - event.record['dateTime']
        if delta > event.record['interval'] * 60:
            logdbg("Skipping record: time difference %s too big" % delta)
            return
        self.save_data(self.get_data())
        if self.max_age is not None:
            self.prune_data(now - self.max_age)

    def save_data(self, record):
        """save data to database"""
        self.archive.addRecord(record)

    def prune_data(self, ts):
        """delete records with dateTime older than ts"""
        sql = "delete from %s where dateTime < %d" % (self.table, ts)
        self.archive.getSql(sql)
        try:
            # sqlite databases need some help to stay small
            self.archive.getSql('vacuum')
        except Exception, e:
            pass

    COLUMNS = re.compile('[\S]+\s+[\d]+\s+[\d.]+\s+[\d.]+\s+([\d]+)\s+([\d]+)')

    def get_data(self):
        record = {}
        record['dateTime'] = int(time.time()+0.5)  # required by weedb
        record['usUnits'] = weewx.METRIC           # required by weedb
        
        try:
            cmd = 'ps aux'
            p = Popen(cmd, shell=True, stdout=PIPE)
            o = p.communicate()[0]
            for line in o.split('\n'):
                if line.find(self.process) >= 0:
                    m = self.COLUMNS.search(line)
                    if m:
                        record['mem_vsz'] = int(m.group(1))
                        record['mem_rss'] = int(m.group(2))
        except (ValueError, IOError, KeyError), e:
            logerr('apcups_info failed: %s' % e)

        return record


# what follows is a basic unit test of this module.  to run the test:
#
# cd /home/weewx
# PYTHONPATH=bin python bin/user/pmon.py
#
if __name__=="__main__":
    from weewx.wxengine import StdEngine
    config = {}
    config['Station'] = {}
    config['Station']['station_type'] = 'Simulator'
    config['Station']['altitude'] = [0,'foot']
    config['Station']['latitude'] = 0
    config['Station']['longitude'] = 0
    config['Simulator'] = {}
    config['Simulator']['driver'] = 'weewx.drivers.simulator'
    config['Simulator']['mode'] = 'simulator'
    config['ProcessMonitor'] = {}
    config['ProcessMonitor']['database'] = 'cmon_sqlite'
    config['ProcessMonitor']['process'] = 'weewxd'
    config['Databases'] = {}
    config['Databases']['cmon_sqlite'] = {}
    config['Databases']['cmon_sqlite']['root'] = '/tmp'
    config['Databases']['cmon_sqlite']['database'] = 'pmon.sdb'
    config['Databases']['cmon_sqlite']['driver'] = 'weedb.sqlite'
    config['Engines'] = {}
    config['Engines']['WxEngine'] = {}
    config['Engines']['WxEngine']['process_services'] = 'user.pmon.ProcessMonitor'
    engine = StdEngine(config)
    svc = ProcessMonitor(engine, config)
    record = svc.get_data()
    print record

    time.sleep(5)
    record = svc.get_data()
    print record

    time.sleep(5)
    record = svc.get_data()
    print record

    os.remove('/tmp/pmon.sdb')
