#!/usr/bin/python
# $Id: maxbotix.py 1293 2015-03-17 17:03:55Z mwall $
# Copyright 2015 Matthew Wall

# From the maxbotix datasheet:
#
# The MB736X sensors have an RS232 data format (with 0V to Vcc levels) and the
# MB738X sensors have a TTL outputs. The output is an ASCII capital R,
# followed by four ASCII character digits representing the range in
# millimeters, followed by a carriage return (ASCII 13). The maximum range
# reported is 4999 mm (5-meter models) or 9998 mm (10-meter models). A range
# value of 5000 or 9999 corresponds to no target being detected in the field
# of view.
#
# The serial data format is 9600 baud, 8 data bits, no parity, with one stop
# bit (9600-8-N-1).

import serial
import syslog
import time

import weewx.drivers
import weewx.engine
import weewx.units

VERSION = "0.1"

def logmsg(dst, msg):
    syslog.syslog(dst, 'maxbotix: %s' % msg)

def logdbg(msg):
    logmsg(syslog.LOG_DEBUG, msg)

def loginf(msg):
    logmsg(syslog.LOG_INFO, msg)

def logerr(msg):
    logmsg(syslog.LOG_ERR, msg)

def loader(config_dict, engine):
    return MaxbotixDriver(**config_dict['Maxbotix'])


schema = [('dateTime',  'INTEGER NOT NULL UNIQUE PRIMARY KEY'),
          ('usUnits',   'INTEGER NOT NULL'),
          ('interval',  'INTEGER NOT NULL'),
          ('range',     'REAL')]

weewx.units.obs_group_dict['range'] = 'group_range'
weewx.units.obs_group_dict['range2'] = 'group_range'
weewx.units.obs_group_dict['range3'] = 'group_range'
weewx.units.USUnits['group_range'] = 'inch'
weewx.units.MetricUnits['group_range'] = 'cm'
weewx.units.MetricWXUnits['group_range'] = 'cm'


class MaxbotixDriver(weewx.drivers.AbstractDevice):

    def __init__(self, **stn_dict):
        loginf("driver version is %s" % VERSION)
        self.max_tries = int(stn_dict.get('max_tries', 5))
        self.retry_wait = int(stn_dict.get('retry_wait', 10))
        self.poll_interval = float(stn_dict.get('poll_interval', 10))
        loginf("polling interval is %s" % self.poll_interval)
        self.port = stn_dict.get('port', '/dev/ttyUSB1')
        loginf("port is %s" % self.port)

    @property
    def hardware_name(self):
        return "Maxbotix"

    def genLoopPackets(self):
        ntries = 0
        while ntries < self.max_tries:
            ntries += 1
            try:
                with Sensor(self.port) as sensor:
                    v = sensor.get_value()
                ntries = 0
                if v == 5000 or v == 9999:
                    loginf("no target detected: v=%s" % v)
                    v = None
                else:
                    v /= 10.0 # sensor is mm, we want cm
                _packet = {'dateTime': int(time.time() + 0.5),
                           'usUnits': weewx.METRIC,
                           'range': v}
                yield _packet
                time.sleep(self.poll_interval)
            except (serial.serialutil.SerialException, weewx.WeeWxIOError), e:
                logerr("Failed attempt %d of %d to get LOOP data: %s" %
                       (ntries, self.max_tries, e))
                time.sleep(self.retry_wait)
        else:
            msg = "Max retries (%d) exceeded for LOOP data" % self.max_tries
            logerr(msg)
            raise weewx.RetriesExceeded(msg)


class MaxbotixService(weewx.engine.StdService):

    def __init__(self, engine, config_dict):
        loginf("service version is %s" % VERSION)
        self.port = stn_dict.get('port', '/dev/ttyUSB1')
        loginf("port is %s" % self.port)

    def handle_new_loop(self, event):
        self.get_data(event.packet)

    def handle_new_archive(self, event):
        self.get_data(event.record)

    def get_data(self, data):
        try:
            with Sensor(self.port) as sensor:
                data['range'] = sensor.get_value()
        except (serial.serialutil.SerialException, weewx.WeeWxIOError), e:
            logerr("Failed to get reading: %s" % e)


class Sensor():
    def __init__(self, port, baudrate=9600, timeout=3):
        self.port = port
        self.baudrate = baudrate
        self.timeout = timeout
        self.serial_port = None

    def __enter__(self):
        self.open()
        return self

    def __exit__(self, _, value, traceback):
        self.close()

    def open(self):
        logdbg("open serial port %s" % self.port)
        self.serial_port = serial.Serial(self.port, self.baudrate,
                                         timeout=self.timeout)

    def close(self):
        if self.serial_port is not None:
            logdbg("close serial port %s" % self.port)
            self.serial_port.close()
            self.serial_port = None

    def get_value(self):
        b = []
        bad_byte = False
        while True:
            c = self.serial_port.read(1)
            if c == '\r':
                break
            elif c == 'R':
                b = []
            else:
                b.append(c)
        if len(b) != 4:
            raise weewx.WeeWxIOError("Got %d bytes, expected 4" % len(b))
        s = ''.join(b)
        try:
            v = int(s)
        except ValueError, e:
            raise weewx.WeeWxIOError("Bogus value: %s" % e)
        return v


# To test this driver, do the following:
#   PYTHONPATH=/home/weewx/bin python /home/weewx/bin/user/maxbotix.py
if __name__ == "__main__":
    usage = """%prog [options] [--help]"""

    def main():
        import optparse
        syslog.openlog('wee_maxbotix', syslog.LOG_PID | syslog.LOG_CONS)
        parser = optparse.OptionParser(usage=usage)
        parser.add_option('--port', dest="port", metavar="PORT",
                          default='/dev/ttyUSB0',
                          help="The port to use. Default is '/dev/ttyUSB0'.")
        parser.add_option('--test-sensor', dest='tc', action='store_true',
                          help='test the sensor')
        parser.add_option('--test-driver', dest='td', action='store_true',
                          help='test the driver')
        parser.add_option('--test-service', dest='ts', action='store_true',
                          help='test the service')
        (options, args) = parser.parse_args()

        if options.tc:
            test_sensor(options.port)
        elif options.td:
            test_driver()
        elif options.ts:
            test_service(options.port)

    def test_driver():
        import weeutil.weeutil
        driver = MaxbotixDriver()
        for pkt in driver.genLoopPackets():
            print weeutil.weeutil.timestamp_to_string(pkt['dateTime']), pkt

    def test_service(port):
        config = {
            'Station': {
                'station_type': 'Simulator',
                'altitude': [0, 'foot'],
                'latitude': 0,
                'longitude': 0},
            'Simulator': {
                'driver': 'weewx.drivers.simulator',
                'mode': 'simulator'},
            'Maxbotix': {
                'port': port},
            'Engine': {
                'Services': {
                    'archive_services': 'user.maxbotix.MaxbotixService'}}}
        engine = weewx.engine.StdEngine(config)
        svc = MaxbotixService(engine, config)
        while True:
            data = dict()
            svc.get_data(data)
            print data
            time.sleep(10)

    def test_sensor(port):
        with Sensor(port) as sensor:
            while True:
                print sensor.get_value()
                time.sleep(10)

    main()
