#!/bin/env python
# $Id$

import configobj
import math
import os
import shutil
import string
import sys
import time
import unittest

import weewx
import weewx.manager
import weewx.reportengine
import weewx.station


# FIXME: these belong in a common testutil module
class TestUtil(object):
    def __init__(self, subdir, tmpdir='/var/tmp/weewx_test'):
        self.subdir = subdir
        self.tmpdir = tmpdir

    @staticmethod
    def rmdir(d):
        try:
            os.rmdir(d)
        except:
            pass

    @staticmethod
    def rmtree(d):
        try:
            shutil.rmtree(d)
        except:
            pass

    @staticmethod
    def mkdir(d):
        try:
            os.makedirs(d)
        except:
            pass

    @staticmethod
    def rmfile(name):
        try:
            os.remove(name)
        except:
            pass

    def get_tmpdir(self):
        return self.tmpdir + '/' + self.subdir

    def get_testdir(self, name):
        return self.get_tmpdir() + '/' + name


class FakeData(object):
    '''generate fake data for testing. portions copied from gen_fake_data.py'''

    start_tt = (2010,1,1,0,0,0,0,0,-1)
    stop_tt  = (2010,2,2,0,0,0,0,0,-1)
    start_ts = int(time.mktime(start_tt))
    stop_ts  = int(time.mktime(stop_tt))
    interval = 600

    @staticmethod
    def create_weather_database(config_dict,
                                start_ts=start_ts, stop_ts=stop_ts,
                                interval=interval, units=weewx.US):
        with weewx.manager.open_database(config_dict,
                                         data_binding='wx_binding',
                                         initialize=True) as dbm:
            dbm.addRecord(FakeData.gen_fake_data(start_ts, stop_ts, interval,
                                                 units))
            dbm.backfill_day_summary()

    @staticmethod
    def gen_fake_data(start_ts=start_ts, stop_ts=stop_ts, interval=interval,
                      units=weewx.US):
        # Four day weather cycle:
        weather_cycle = 3600*24.0*4

        if units == weewx.METRIC:
            daily_temp_range = 30.0
            annual_temp_range = 60.0
            avg_temp = 5.0
            weather_baro_range = 68.0
            weather_wind_range = 16.0
            avg_baro = 1015.0
            rain = 0.2
        else:
            daily_temp_range = 40.0
            annual_temp_range = 80.0
            avg_temp = 40.0
            weather_baro_range = 2.0
            weather_wind_range = 10.0
            avg_baro = 30.0
            rain = 0.02

        count = 0
        for ts in xrange(start_ts, stop_ts+interval, interval):
            daily_phase  = (ts - start_ts) * 2.0 * math.pi / (3600*24.0)
            annual_phase = (ts - start_ts) * 2.0 * math.pi / (3600*24.0*365.0)
            weather_phase= (ts - start_ts) * 2.0 * math.pi / weather_cycle
            record = {}
            record['dateTime']  = ts
            record['usUnits']   = units
            record['interval']  = interval
            record['outTemp']   = 0.5 * (-daily_temp_range*math.sin(daily_phase) - annual_temp_range*math.cos(annual_phase)) + avg_temp
            record['barometer'] = 0.5 * weather_baro_range*math.sin(weather_phase) + avg_baro
            record['windSpeed'] = abs(weather_wind_range*(1.0 + math.sin(weather_phase)))
            record['windDir'] = math.degrees(weather_phase) % 360.0
            record['windGust'] = 1.2*record['windSpeed']
            record['windGustDir'] = record['windDir']
            if math.sin(weather_phase) > .95:
                record['rain'] = rain if math.sin(weather_phase) > 0.98 else 0.01
            else:
                record['rain'] = 0.0

        # Make every 71st observation (a prime number) a null. This is a
        # deterministic algorithm, so it will produce the same results every
        # time.                             
            for obs_type in filter(lambda x : x not in ['dateTime', 'usUnits', 'interval'], record):
                count+=1
                if count%71 == 0:
                    record[obs_type] = None
            yield record


class TestConfig(object):
    metric_unit_groups = """
        group_altitude     = meter
        group_degree_day   = degree_C_day
        group_direction    = degree_compass
        group_moisture     = centibar
        group_percent      = percent
        group_pressure     = inHg
        group_radiation    = watt_per_meter_squared
        group_rain         = mm
        group_rainrate     = mm_per_hour
        group_speed        = km_per_hour
        group_speed2       = km_per_hour2
        group_temperature  = degree_C
        group_uv           = uv_index
        group_volt         = volt
"""

    us_unit_groups = """
        group_altitude     = foot
        group_degree_day   = degree_F_day
        group_direction    = degree_compass
        group_moisture     = centibar
        group_percent      = percent
        group_pressure     = mbar
        group_radiation    = watt_per_meter_squared
        group_rain         = inch
        group_rainrate     = inch_per_hour
        group_speed        = mile_per_hour
        group_speed2       = knot2
        group_temperature  = degree_F
        group_uv           = uv_index
        group_volt         = volt
"""

    # FIXME: make weewx work without having to specify so many items in config
    #   Units
    #   Labels
    #   archive/stats databases
    skin_contents = """
[Units]
    [[Groups]]
        GROUPS

        # The following groups are used internally and should not be changed:
        group_count        = count
        group_interval     = minute
        group_time         = unix_epoch
        group_elapsed      = second

    [[StringFormats]]
        centibar           = %.0f
        cm                 = %.2f
        cm_per_hour        = %.2f
        degree_C           = %.1f
        degree_F           = %.1f
        degree_compass     = %.0f
        foot               = %.0f
        hPa                = %.1f
        inHg               = %.3f
        inch               = %.2f
        inch_per_hour      = %.2f
        km_per_hour        = %.0f
        km_per_hour2       = %.1f
        knot               = %.0f
        knot2              = %.1f
        mbar               = %.1f
        meter              = %.0f
        meter_per_second   = %.1f
        meter_per_second2  = %.1f
        mile_per_hour      = %.1f
        mile_per_hour2     = %.1f
        mm                 = %.1f
        mmHg               = %.1f
        mm_per_hour        = %.1f
        percent            = %.0f
        uv_index           = %.1f
        volt               = %.1f
        watt_per_meter_squared = %.0f
        NONE               = "    -"

    [[Labels]]
        centibar          = " cb"
        cm                = " cm"
        cm_per_hour       = " cm/hr"
        degree_C          =   C
        degree_F          =   F
        degree_compass    =   deg
        foot              = " feet"
        hPa               = " hPa"
        inHg              = " inHg"
        inch              = " in"
        inch_per_hour     = " in/hr"
        km_per_hour       = " kph"
        km_per_hour2      = " kph"
        knot              = " knots"
        knot2             = " knots"
        mbar              = " mbar"
        meter             = " meters"
        meter_per_second  = " m/s"
        meter_per_second2 = " m/s"
        mile_per_hour     = " mph"
        mile_per_hour2    = " mph"
        mm                = " mm"
        mmHg              = " mmHg"
        mm_per_hour       = " mm/hr"
        percent           =   %
        volt              = " V"
        watt_per_meter_squared = " W/m^2"
        NONE              = ""
        
    [[TimeFormats]]
        day        = %H:%M
        week       = %H:%M on %A
        month      = %d.%m.%Y %H:%M
        year       = %d.%m.%Y %H:%M
        rainyear   = %d.%m.%Y %H:%M
        current    = %d.%m.%Y %H:%M
        ephem_day  = %H:%M
        ephem_year = %d.%m.%Y %H:%M
    [[DegreeDays]]
[Labels]
[Almanac]
    moon_phases = n,wc,fq,wg,f,wg,lq,wc
[CheetahGenerator]
    search_list = heatmap.HeatmapVariables
    encoding = html_entities
    [[ToDate]]
        [[[current]]]
            template = index.html.tmpl
[Generators]
    generator_list = weewx.cheetahgenerator.CheetahGenerator
"""

    @staticmethod
    def create_config(test_dir, skin_dir):
        cd = configobj.ConfigObj()
        cd['debug'] = 1
        cd['WEEWX_ROOT'] = test_dir
        cd['Station'] = {
            'station_type': 'Simulator',
            'altitude': [10,'foot'],
            'latitude': 42.358,
            'longitude': -71.106
            }
        cd['Simulator'] = {
            'driver': 'weewx.drivers.simulator',
            'mode': 'generator'
            }
        cd['DataBindings'] = {
            'wx_binding': {
                'database': 'wx_sqlite'
                }
            }
        cd['Databases'] = {
            'wx_sqlite' : {
                'root': '%(WEEWX_ROOT)s',
                'database_name': test_dir + '/weewx.sdb',
                'driver': 'weedb.sqlite'
                }
            }
        cd['StdReport'] = {
            'HTML_ROOT': test_dir + '/html',
            'SKIN_ROOT': test_dir,
            'TestReport': { 'skin' : skin_dir }
            }
        cd['StdArchive'] = {
            'data_binding': 'wx_binding'
            }
        return cd

    @staticmethod
    def create_skin_conf(test_dir, skin_dir, units=weewx.US):
        """create minimal skin config file for testing"""
        groups = TestConfig.metric_unit_groups if units == weewx.METRIC else TestConfig.us_unit_groups
        contents = TestConfig.skin_contents.replace('GROUPS', groups)
        TestUtil.mkdir(test_dir + '/' + skin_dir)
        fn = test_dir + '/' + skin_dir + '/skin.conf'
        f = open(fn, 'w')
        f.write(contents)
        f.close()
        
        
tu = TestUtil('test_heatmap')


class HeatmapTest(unittest.TestCase):
    def setup_template_test(self, tname, tmpl):
        tdir = tu.get_testdir(tname)
        tu.rmtree(tdir)

        skin_dir = 'testskin'
        cd = TestConfig.create_config(tdir, skin_dir)
        FakeData.create_weather_database(cd)
        TestConfig.create_skin_conf(tdir, skin_dir)

        ts = int(time.mktime((2013,8,22,12,0,0,0,0,-1)))
        stn_info = weewx.station.StationInfo(**cd['Station'])
        t = weewx.reportengine.StdReportEngine(cd, stn_info, ts)

        shutil.copyfile('skins/heatmap/heatmap_table.inc',
                        tdir + '/' + skin_dir + '/heatmap_table.inc')

        fn = tdir + '/' + skin_dir + '/index.html.tmpl'
        f = open(fn, 'w')
        f.write(tmpl)
        f.close()

        return t, tdir
          
    def run_template_test(self, tname, tmpl, expected_lines):
        t, tdir = self.setup_template_test(tname, tmpl)
        t.run()
        self.compare_contents(tdir + '/html/index.html', expected_lines)

    def compare_contents(self, filename, expected):
        expected_lines = string.split(expected, '\n')

        actual = open(filename)
        actual_lines = []
        for actual_line in actual:
            actual_lines.append(actual_line)
        actual.close()
        if len(actual_lines) != len(expected_lines):
            raise AssertionError('wrong number of lines in %s: %d != %d' %
                                 (filename,
                                  len(actual_lines),
                                  len(expected_lines)))

        lineno = 0
        diffs = []
        for actual_line in actual_lines:
            try:
                self.assertEqual(string.rstrip(actual_line),
                                 expected_lines[lineno])
            except AssertionError, e:
                diffs.append('line %d: %s' % (lineno+1, e))
            lineno += 1
        if len(diffs) > 0:
            raise AssertionError('differences found in %s:\n%s' %
                                 (filename, '\n'.join(diffs)))

    # FIXME: i would prefer to do a function, but this does not work, even
    # when we use the staticmethod hack:
    #
    # $heatmap_table($s_ts, $e_ts, $data_avg)
    #
    def test_map(self):
        self.run_template_test('test_map',
                               """<html>
  <head>
    <style>
.heatmap {
  width: 500px;
}
    </style>
  </head>
  <body>
#set $s_ts,$e_ts,$min_value,$max_value,$data_min,$data_avg,$data_max = $heatmap.get_data('outTemp')

#set global $start_ts = $s_ts
#set global $end_ts = $e_ts
#set global $min_val = $min_value
#set global $max_val = $max_value

#set global $data = $data_avg
#include 'heatmap_table.inc'

#set global $data = $data_min
#include 'heatmap_table.inc'

#set global $data = $data_max
#include 'heatmap_table.inc'
  </body>
</html>
""",
                               """<html>
  <body>
  </body>
</html>
""")

            
# use '--test test_name' to specify a single test
if __name__ == '__main__':
    if len(sys.argv) == 3 and sys.argv[1] == '--test':
        testname = sys.argv[2]
        suite = unittest.TestSuite(map(HeatmapTest, [testname]))
        unittest.TextTestRunner(verbosity=2).run(suite)
    else:
        unittest.main()
