# $Id: cwxn.py 998 2014-08-15 03:16:34Z mwall $
# Copyright 2014 Matthew Wall

# Emit loop data to wxnow.txt file with Cumulus format:
#   http://wiki.sandaysoft.com/a/Wxnow.txt
#
# Put this file in bin/user/cwxn.py, then add this to your weewx.conf:
#
# [CumulusWXNow]
#     filename = /path/to/wxnow.txt
#
# [Engines]
#     [[WxEngine]]
#         process_services = ..., user.cwxn.CumulusWXNow
#

# FIXME: when value is None, we insert a 0.  but is there something in the
#        aprs spec that is more appropriate?

import math
import time
import syslog

import weewx
import weewx.wxformulas
import weeutil.weeutil
import weeutil.Sun
from weewx.wxengine import StdService

VERSION = 'X'

def convert(v, metric, group, from_unit_system, to_units):
    ut = weewx.units.getStandardUnitType(from_unit_system, metric)
    vt = (v, ut[0], group)
    v = weewx.units.convert(vt, to_units)[0]
    return v

def nullproof(key, data):
    if key in data and data[key] is not None:
        return data[key]
    return 0

def calcRainHour(archive, ts):
    sts = ts - 3600
    val = archive.getSql("SELECT SUM(rain) FROM archive "
                         "WHERE dateTime>? AND dateTime<=?",
                         (sts, ts))
    if val is None:
        return None
    return val[0]

class CumulusWXNow(StdService):

    def __init__(self, engine, config_dict):
        super(CumulusWXNow, self).__init__(engine, config_dict)
        try:
            db = config_dict['StdArchive']['archive_database']
            self.database_dict = config_dict['Databases'][db]
        except KeyError, e:
            syslog.syslog(syslog.LOG_DEBUG, 
                          "cwxn: cannot determine database configuration: " % e)
            return
        d = config_dict.get('CumulusWXNow', {})
        self.filename = d.get('filename', '/var/tmp/wxnow.txt')
        self.bind(weewx.NEW_LOOP_PACKET, self.handle_new_loop)

    def handle_new_loop(self, event):
        try:
            data = {}
            with weewx.archive.Archive.open(self.database_dict) as archive:
                data = self.calculate(event.packet, archive)
            self.write_data(data)
        except Exception, e:
            weeutil.weeutil.log_traceback('cwxn: **** ')

    def calculate(self, packet, archive):
        pu = packet.get('usUnits')
        data = {}
        data['dateTime'] = packet['dateTime']
        data['windDir'] = nullproof('windDir', packet)
        v = nullproof('windSpeed', packet)
        data['windSpeed'] = convert(v, 'windSpeed', 'group_speed', pu, 'mile_per_hour')
        v = nullproof('windGust', packet)
        data['windGust'] = convert(v, 'windGust', 'group_speed', pu, 'mile_per_hour')
        v = nullproof('outTemp', packet)
        data['outTemp'] = convert(v, 'outTemp', 'group_temperature', pu, 'degree_F')
        v = calcRainHour(archive, data['dateTime'])
        if v is None:
            v = 0
        data['hourRain'] = convert(v, 'rain', 'group_rain', pu, 'inch')
        v = nullproof('rain24', packet)
        data['rain24'] = convert(v, 'rain', 'group_rain', pu, 'inch')
        v = nullproof('dayRain', packet)
        data['dayRain'] = convert(v, 'rain', 'group_rain', pu, 'inch')
        data['outHumidity'] = nullproof('outHumidity', packet)
        v = nullproof('barometer', packet)
        data['barometer'] = convert(v, 'pressure', 'group_pressure', pu, 'mbar')
        return data

    def write_data(self, data):
        fields = []
        fields.append("%03d" % int(data['windDir']))
        fields.append("/%03d" % int(data['windSpeed']))
        fields.append("g%03d" % int(data['windGust']))
        fields.append("t%03d" % int(data['outTemp']))
        fields.append("r%03d" % int(data['hourRain'] * 100))
        fields.append("p%03d" % int(data['rain24'] * 100))
        fields.append("P%03d" % int(data['dayRain'] * 100))
        if data['outHumidity'] < 0 or 100 <= data['outHumidity']:
            data['outHumidity'] = 0
        fields.append("h%03d" % int(data['outHumidity']))
        fields.append("b%05d" % int(data['barometer'] * 10))
        with open(self.filename, 'w') as f:
            f.write(time.strftime("%b %d %Y %H:%M\n",
                                  time.localtime(data['dateTime'])))
            f.write(''.join(fields))
            f.write("\n")
