# $Id: csv.py 1748 2017-10-06 01:45:25Z mwall $
# Copyright 2015 Matthew Wall

import os
import os.path
import time

import weewx
import weewx.engine
import weeutil.weeutil

VERSION = "0.7"

def logmsg(level, msg):
    syslog.syslog(level, 'csv: %s' % msg)

def logdbg(msg):
    logmsg(syslog.LOG_DEBUG, msg)

def loginf(msg):
    logmsg(syslog.LOG_INFO, msg)

def logerr(msg):
    logmsg(syslog.LOG_ERR, msg)

class CSV(weewx.engine.StdService):
    def __init__(self, engine, config_dict):
        super(CSV, self).__init__(engine, config_dict)
        loginf("service version is %s" % VERSION)
        d = config_dict.get('CSV', {})
        self.filename = d.get('filename', '/var/tmp/data.csv')
        self.emit_header = weeutil.weeutil.to_bool(d.get('header', True))
        self.mode = d.get('mode', 'append')
        self.append_datestamp = weeutil.weeutil.to_bool(d.get('append_datestamp', True))
        self.binding = d.get('binding', 'loop')
        if self.binding == 'loop':
            self.bind(weewx.NEW_LOOP_PACKET, self.handle_new_loop)
        else:
            self.bind(weewx.NEW_ARCHIVE_RECORD, self.handle_new_archive)

    def handle_new_loop(self, event):
        self.write_data(event.packet)

    def handle_new_archive(self, event):
        self.write_data(event.record)

    def write_data(self, data):
        flag = "a" if self.mode == 'append' else "w"
        filename = self.filename
        if self.append_datestamp:
            basename = filename
            ext = ''
            idx = filename.find('.')
            if idx > -1:
                basename = filename[:idx]
                ext = filename[idx:]
            tstr = time.strftime('%Y-%m', time.gmtime(data['dateTime']))
            filename = "%s-%s%s" % (basename, tstr, ext)
        header = None
        if self.emit_header and (
            not os.path.exists(filename) or flag == "w"):
            header = '# %s\n' % ','.join(self.sort_keys(data))
        with open(filename, flag) as f:
            if header:
                f.write(header)
            f.write('%s\n' % ','.join(self.sort_data(data)))

    def sort_keys(self, record):
        fields = ['dateTime']
        for k in sorted(record):
            if k != 'dateTime':
                fields.append(k)
        return fields

    def sort_data(self, record):
        fields = [str(record['dateTime'])]
        for k in sorted(record):
            if k != 'dateTime':
                fields.append(str(record[k]))
        return fields
