# $Id: crt.py 899 2014-05-03 12:30:06Z mwall $
# Copyright 2013 Matthew Wall

# Emit loop data to file in Cumulus realtime format.
#   http://wiki.sandaysoft.com/a/Realtime.txt
#
# Put this file in bin/user/crt.py, then add this to your weewx.conf:
#
# [CumulusRealTime]
#     date_separator = /
#     filename = /path/to/realtime.txt
#
# [Engines]
#     [[WxEngine]]
#         process_services = ..., user.crt.CumulusRealTime
#
# Note that most of the code in this file is to calculate/lookup data that
# are not directly provided by weewx in a LOOP packet.
#
# The cumulus 'specification' for realtime.txt is ambiguous:
#
#   pressure trend interval is not specified, we use 3 hours for pressure
#   temperature trend interval is not specified, we use 3 hours for temperature
#   wind avg speed/dir interval is not specified, we use 10 minutes
#   time zone is not specified, we use utc everywhere
#   pressure is not specified, we use 'barometer' not 'pressure' or 'altimeter'
#
#   FIXME: what if None?  we return NULL, but cumulus spec does not specify
#   FIXME: is_sunny is ill-defined.  what radiation threshold indicates true?
#   FIXME: ensure day/month/hour/minute/second is prefixed with leading zero
#   FIXME: max_rad is not yet implemented

import math
import time

import weewx
import weewx.wxformulas
import weeutil.weeutil
import weeutil.Sun
from weewx.wxengine import StdService

COMPASS_POINTS = ['N','NNE','NE','ENE','E','ESE','SE','SSE',
                  'S','SSW','SW','WSW','W','WNW','NW','NNW','N']

# map weewx unit names to cumulus unit names
UNITS_WIND = {'mile_per_hour':      'mph',
              'meter_per_second':   'm/s',
              'kilometer_per_hour': 'km/h',
              'knot':               'kts'}
UNITS_TEMP = {'degree_C': 'C',
              'degree_F': 'F'}
UNITS_PRES = {'inHg': 'in',
              'mbar': 'mb',
              'hPa':  'hPa'}
UNITS_RAIN = {'in': 'in',
              'mm': 'mm'}

def getAltitudeFt(config_dict):
    """Get the altitude, in meters, from the Station section of the dict."""
    altitude_t = weeutil.weeutil.option_as_list(
        config_dict['Station'].get('altitude', (None, None)))
    altitude_vt = (float(altitude_t[0]), altitude_t[1], "group_altitude")
    altitude_f = weewx.units.convert(altitude_vt, 'foot')[0]
    return altitude_f

def getLatitude(config_dict):
    return float(config_dict['Station'].get('latitude', None))

def getLongitude(config_dict):
    return float(config_dict['Station'].get('longitude', None))

def degreeToCompass(x):
    if x is None:
        return None
    idx = int((x + 11.25) / 22.5)
    return COMPASS_POINTS[idx]

# http://www.spc.noaa.gov/faq/tornado/beaufort.html
def knotToBeaufort(x):
    if x is None:
        return None
    if x < 1:
        return 0  # calm
    elif x < 4:
        return 1  # light air
    elif x < 7:
        return 2  # light breeze
    elif x < 11:
        return 3  # gentle breeze
    elif x < 17:
        return 4  # moderate breeze
    elif x < 22:
        return 5  # fresh breeze
    elif x < 28:
        return 6  # strong breeze
    elif x < 34:
        return 7  # near gale
    elif x < 41:
        return 8  # gale
    elif x < 48:
        return 9  # strong gale
    elif x < 56:
        return 10 # storm
    elif x < 64:
        return 11 # violent storm
    return 12 # hurricane

def calcAvgWindSpeed(archive, ts, interval=600):
    sts = ts - interval
    val = archive.getSql("SELECT AVG(windSpeed) FROM archive "
                         "WHERE dateTime>? AND dateTime<=?",
                         (sts, ts))
    return val[0]

def calcAvgWindDir(archive, ts, interval=600):
    sts = ts - interval
    val = archive.getSql("SELECT AVG(windDir) FROM archive "
                         "WHERE dateTime>? AND dateTime<=?",
                         (sts, ts))
    return val[0]

def calc10MinHighGust(archive, ts):
    sts = ts - 600
    val = archive.getSql("SELECT MAX(windGust) FROM archive "
                         "WHERE dateTime>? AND dateTime<=?",
                         (sts, ts))
    return val[0]

def calc10MinAvgWindDir(archive, ts):
    sts = ts - 600
    val = archive.getSql("SELECT AVG(windGust) FROM archive "
                         "WHERE dateTime>? AND dateTime<=?",
                         (sts, ts))
    return val[0]

# calculate wind run since midnight
def calcWindRun(archive, ts):
    run = 0
    sod_ts = weeutil.weeutil.startOfDay(ts)
    for row in archive.genSql("SELECT interval,windSpeed FROM archive "
                              "WHERE dateTime>? AND dateTime<=?", 
                              (sod_ts, ts)):
        run += row[1] * (row[0] / 60)
    return run

# calculate trend over past n hours, default to 3 hour window
def calcTrend(label, new_val, archive, ts, n=3):
    lastts = ts - n * 3600
    qstr = "SELECT %s FROM archive WHERE dateTime>? AND dateTime<=?" % label
    old_val = archive.getSql(qstr, (lastts, ts))
    return new_val - old_val[0]

def calcRainHour(archive, ts):
    sts = ts - 3600
    val = archive.getSql("SELECT SUM(rain) FROM archive "
                         "WHERE dateTime>? AND dateTime<=?",
                         (sts, ts))
    return val[0]

def calcRainMonth(archive, ts):
    span = weeutil.weeutil.archiveMonthSpan(ts)
    val = archive.getSql("SELECT SUM(rain) FROM archive "
                         "WHERE dateTime>? AND dateTime<=?",
                         (span.start, ts))
    return val[0]

def calcRainYear(archive, ts):
    span = weeutil.weeutil.archiveYearSpan(ts)
    val = archive.getSql("SELECT SUM(rain) FROM archive "
                         "WHERE dateTime>? AND dateTime<=?",
                         (span.start, ts))
    return val[0]

def calcRainYesterday(archive, ts):
    ts = weeutil.weeutil.startOfDay(ts)
    sts = ts - 3600*24;
    val = archive.getSql("SELECT SUM(rain) FROM archive "
                         "WHERE dateTime>? AND dateTime<=?",
                         (sts, ts))
    return val[0]

def calcETToday(archive, ts):
    sts = weeutil.weeutil.startOfDay(ts)
    val = archive.getSql("SELECT SUM(ET) FROM archive "
                         "WHERE dateTime>? AND dateTime<=?",
                         (sts, ts))
    return val[0]

def calcMinMax(label, archive, ts, minmax='MAX'):
    sts = weeutil.weeutil.startOfDay(ts)
    qstr = "SELECT %s(%s) FROM archive WHERE dateTime>=? AND dateTime<=?" % (minmax,label)
    val = archive.getSql(qstr, (sts, ts))
    qstr = "SELECT dateTime FROM archive WHERE dateTime>=? AND dateTime<=? AND %s=?" % label
    t = archive.getSql(qstr, (sts, ts, val[0]))
    tstr = time.strftime("%H:%M", time.gmtime(t[0]))
    return val[0],tstr

def calcHumidex(t_C, dewpoint_K):
    v = 1/273.16
    v -= 1/dewpoint_K
    v *= 5417.7530
    v = math.exp(v)
    v *= 6.11
    v -= 10
    v *= 0.5555
    return t_C + v

def calcIsDaylight(ts_utc, lat, lon):
    x = weeutil.weeutil.startOfDayUTC(ts_utc)
    x_tt = time.gmtime(x)
    y, m, d = x_tt[:3]
    (sunrise_utc, sunset_utc) = weeutil.Sun.sunRiseSet(y, m, d, lon, lat)
    if sunrise_utc < ts_utc < sunset_utc:
        return 1
    return 0

def calcDaylightHours(ts_utc, lat, lon):
    x = weeutil.weeutil.startOfDayUTC(ts_utc)
    x_tt = time.gmtime(x)
    y, m, d = x_tt[:3]
    (sunrise_utc, sunset_utc) = weeutil.Sun.sunRiseSet(y, m, d, lon, lat)
    if ts_utc <= sunrise_utc:
        return 0
    elif ts_utc < sunset_utc:
        return (ts_utc - sunrise_utc) / 60
    return (sunset_utc - sunrise_utc) / 60

def calcCloudBase(outTemp_F, dewpoint_F, altitude_foot):
    return int((outTemp_F - dewpoint_F) / 4.4 * 1000) + altitude_foot

# this depends on the weather station
#
# Vantage
#   packet['rxCheckPercent'] == 0
#
# FineOffset
#   packet['status'] & 0x40
#
# TE923
#   packet['sensorX_state'] == STATE_MISSING_LINK
#   packet['wind_state'] == STATE_MISSING_LINK
#   packet['rain_state'] == STATE_MISSING_LINK
#   packet['uv_state'] == STATE_MISSING_LINK
#
# WMR100
# WMR200
# WMR9x8
#
# WS28xx
#   packet['rxCheckPercent'] == 0
#
# WS23xx
#   packet['cn'] == 'lost'
#
def lostSensorContact():
    # FIXME: implement lostSensorContact
    return 0

class CumulusRealTime(StdService):

    def __init__(self, engine, config_dict):
        super(CumulusRealTime, self).__init__(engine, config_dict)
        try:
            db = config_dict['StdArchive']['archive_database']
            self.database_dict = config_dict['Databases'][db]
        except KeyError, e:
            syslog.syslog(syslog.LOG_DEBUG, 
                          "crt: cannot determine database configuration: " % e)
            return
        self.altitude_ft = getAltitudeFt(config_dict)
        self.latitude = getLatitude(config_dict)
        self.longitude = getLongitude(config_dict)

        d = config_dict.get('CumulusRealTimeService', {})
        self.filename = d.get('filename', '/var/tmp/realtime.txt')
        self.datesep = d.get('date_separator', '/')
        self.bind(weewx.NEW_LOOP_PACKET, self.handle_new_loop)

    def handle_new_loop(self, event):
        data = {}
        with weewx.archive.Archive.open(self.database_dict) as archive:
            data = self.calculate(event.packet, archive)
        self.write_data(data)

    def write_data(self, data):
        with open(self.filename, 'w') as f:
            f.write(self.create_realtime_string(data))
            f.write("\n")

    # calculate the data elements that cumulus needs that weewx does not
    # provide directly.
    def calculate(self, packet, archive):
        ts = packet.get('dateTime')
        pu = packet.get('usUnits')
        data = dict(packet)
        data['windSpeed_avg'] = calcAvgWindSpeed(archive, ts)
        data['windDir_compass'] = degreeToCompass(packet.get('windDir'))
        x = packet.get('windSpeed')
        ut = weewx.units.getStandardUnitType(pu, 'windSpeed')
        vt = (x, ut[0], 'group_speed')
        v = weewx.units.convert(vt, 'knot')[0]
        data['windSpeed_beaufort'] = knotToBeaufort(v)
        data['units_wind'] = UNITS_WIND.get(ut[0])
        ut = weewx.units.getStandardUnitType(pu, 'outTemp')
        data['units_temperature'] = UNITS_TEMP.get(ut[0],ut[0])
        ut = weewx.units.getStandardUnitType(pu, 'barometer')
        data['units_pressure'] = UNITS_PRES.get(ut[0],ut[0])
        ut = weewx.units.getStandardUnitType(pu, 'rain')
        data['units_rain'] = UNITS_RAIN.get(ut[0],ut[0])
        data['wind_run'] = calcWindRun(archive, ts)
        data['pressure_trend'] = calcTrend('barometer', packet.get('barometer'),
                                           archive, ts)
        data['temperature_trend'] = calcTrend('outTemp', packet.get('outTemp'),
                                              archive, ts)
        data['rain_month'] = calcRainMonth(archive, ts)
        data['rain_year'] = calcRainYear(archive, ts)
        data['rain_yesterday'] = calcRainYesterday(archive, ts)
        v,t = calcMinMax('outTemp', archive, ts, 'MAX')
        data['outTemp_max'] = v
        data['outTemp_max_time'] = t
        v,t = calcMinMax('outTemp', archive, ts, 'MIN')
        data['outTemp_min'] = v
        data['outTemp_min_time'] = t
        v,t = calcMinMax('windSpeed', archive, ts, 'MAX')
        data['windSpeed_max'] = v
        data['windSpeed_max_time'] = t
        v,t = calcMinMax('windGust', archive, ts, 'MAX')
        data['windGust_max'] = v
        data['windGust_max_time'] = t
        v,t = calcMinMax('barometer', archive, ts, 'MAX')
        data['pressure_max'] = v
        data['pressure_max_time'] = t
        v,t = calcMinMax('barometer', archive, ts, 'MIN')
        data['pressure_min'] = v
        data['pressure_min_time'] = t
        data['10min_high_gust'] = calc10MinHighGust(archive, ts)
        ut = weewx.units.getStandardUnitType(pu, 'outTemp')
        vt = (packet.get('outTemp'), ut[0], 'group_temperature')
        v = weewx.units.convert(vt, 'degree_F')[0]
        x = weewx.wxformulas.heatindexF(v, packet.get('outHumidity'))
        vt = (x, 'degree_F', 'group_temperature')
        v = weewx.units.convert(vt, ut[0])[0]
        data['heatindex'] = v
        ut = weewx.units.getStandardUnitType(pu, 'outTemp')
        vt = (packet.get('outTemp'), ut[0], 'group_temperature')
        t_C = weewx.units.convert(vt, 'degree_C')[0]
        vt = (packet.get('dewpoint'), ut[0], 'group_temperature')
        dp_C = weewx.units.convert(vt, 'degree_C')[0]
        dp_K = dp_C + 273.15
        v = calcHumidex(t_C, dp_K);
        vt = (v, 'degree_C', 'group_temperature')
        v = weewx.units.convert(vt, ut[0])[0]
        data['humidex'] = v
        data['ET_today'] = calcETToday(archive, ts)
        data['10min_avg_wind_bearing'] = calc10MinAvgWindDir(archive, ts)
        data['rain_hour'] = calcRainHour(archive, ts)
        data['is_daylight'] = calcIsDaylight(ts, self.latitude, self.longitude)
        data['lost_sensors_contact'] = lostSensorContact()
        data['avg_wind_dir'] = degreeToCompass(calcAvgWindDir(archive, ts))
        ut = weewx.units.getStandardUnitType(pu, 'outTemp')
        vt = (packet.get('outTemp'), ut[0], 'group_temperature')
        t_F = weewx.units.convert(vt, 'degree_F')[0]
        ut = weewx.units.getStandardUnitType(pu, 'outTemp')
        vt = (packet.get('dewpoint'), ut[0], 'group_temperature')
        d_F = weewx.units.convert(vt, 'degree_F')[0]
        data['cloud_base'] = calcCloudBase(t_F, d_F, self.altitude_ft)
        data['units_cloud_base'] = 'ft'
        if (v < 50):
            ut = weewx.units.getStandardUnitType(pu, 'outTemp')
            vt = (packet.get('outTemp'), ut[0], 'group_temperature')
            t_F = weewx.units.convert(vt, 'degree_F')[0]
            ut = weewx.units.getStandardUnitType(pu, 'windSpeed')
            vt = (packet.get('windSpeed'), ut[0], 'group_speed')
            ws_mph = weewx.units.convert(vt, ut[0])[0]
            x = weewx.wxformulas.windchillF(t_F, ws_mph)
            vt = (x, 'degree_F', 'group_temperature')
            ut = weewx.units.getStandardUnitType(pu, 'outTemp')
            v = weewx.units.convert(vt, ut[0])[0]
        else:
            v = data['heatindex']
        data['apparent_temperature'] = v
        data['sunshine_hours'] = calcDaylightHours(ts, self.latitude, self.longitude)
        data['max_rad'] = None
        data['is_sunny'] = data['is_daylight']
        return data

    def format(self, data, label):
        value = data.get(label, 'NULL')
        if value is None:
            value = 'NULL'
        return str(value)

    # the * indicates a field that is not part of a typical LOOP packet
    # the ## indicates calculation is not yet implemented
    def create_realtime_string(self, data):
        fields = []
        datefmt = "%%d%s%%m%s%%y" % (self.datesep, self.datesep)
        tstr = time.strftime(datefmt, time.gmtime(data['dateTime']))
        fields.append(tstr)                                        # 1
        tstr = time.strftime("%H:%M:%S", time.gmtime(data['dateTime']))
        fields.append(tstr)                                        # 2
        fields.append(self.format(data, 'outTemp'))                # 3
        fields.append(self.format(data, 'outHumidity'))            # 4
        fields.append(self.format(data, 'dewpoint'))               # 5
        fields.append(self.format(data, 'windSpeed_avg'))          # 6  *
        fields.append(self.format(data, 'windSpeed'))              # 7
        fields.append(self.format(data, 'windDir'))                # 8
        fields.append(self.format(data, 'rainRate'))               # 9
        fields.append(self.format(data, 'dayRain'))                # 10
        fields.append(self.format(data, 'barometer'))              # 11
        fields.append(self.format(data, 'windDir_compass'))        # 12 *
        fields.append(self.format(data, 'windSpeed_beaufort'))     # 13 *
        fields.append(self.format(data, 'units_wind'))             # 14 *
        fields.append(self.format(data, 'units_temperature'))      # 15 *
        fields.append(self.format(data, 'units_pressure'))         # 16 *
        fields.append(self.format(data, 'units_rain'))             # 17 *
        fields.append(self.format(data, 'wind_run'))               # 18 *
        fields.append(self.format(data, 'pressure_trend'))         # 19 *
        fields.append(self.format(data, 'rain_month'))             # 20 *
        fields.append(self.format(data, 'rain_year'))              # 21 *
        fields.append(self.format(data, 'rain_yesterday'))         # 22 *
        fields.append(self.format(data, 'inTemp'))                 # 23
        fields.append(self.format(data, 'inHumidity'))             # 24
        fields.append(self.format(data, 'windChill'))              # 25
        fields.append(self.format(data, 'temperature_trend'))      # 26 *
        fields.append(self.format(data, 'outTemp_max'))            # 27 *
        fields.append(self.format(data, 'outTemp_max_time'))       # 28 *
        fields.append(self.format(data, 'outTemp_min'))            # 29 *
        fields.append(self.format(data, 'outTemp_min_time'))       # 30 *
        fields.append(self.format(data, 'windSpeed_max'))          # 31 *
        fields.append(self.format(data, 'windSpeed_max_time'))     # 32 *
        fields.append(self.format(data, 'windGust_max'))           # 33 *
        fields.append(self.format(data, 'windGust_max_time'))      # 34 *
        fields.append(self.format(data, 'pressure_max'))           # 35 *
        fields.append(self.format(data, 'pressure_max_time'))      # 36 *
        fields.append(self.format(data, 'pressure_min'))           # 37 *
        fields.append(self.format(data, 'pressure_min_time'))      # 38 *
        fields.append('%s' % weewx.__version__)                    # 39
        fields.append('0')                                         # 40
        fields.append(self.format(data, '10min_high_gust'))        # 41 *
        fields.append(self.format(data, 'heatindex'))              # 42 *
        fields.append(self.format(data, 'humidex'))                # 43 *
        fields.append(self.format(data, 'UV'))                     # 44
        fields.append(self.format(data, 'ET_today'))               # 45 *
        fields.append(self.format(data, 'radiation'))              # 46
        fields.append(self.format(data, '10min_avg_wind_bearing')) # 47 *
        fields.append(self.format(data, 'rain_hour'))              # 48 *
        fields.append('0')                                         # 49
        fields.append(self.format(data, 'is_daylight'))            # 50 *
        fields.append(self.format(data, 'lost_sensors_contact'))   # 51 *
        fields.append(self.format(data, 'avg_wind_dir'))           # 52 *
        fields.append(self.format(data, 'cloud_base'))             # 53 *
        fields.append(self.format(data, 'units_cloud_base'))       # 54 *
        fields.append(self.format(data, 'apparent_temperature'))   # 55 *
        fields.append(self.format(data, 'sunshine_hours'))         # 56 *
        fields.append(self.format(data, 'max_rad'))                # 57 ##
        fields.append(self.format(data, 'is_sunny'))               # 58 *
        return ' '.join(fields)
