# $Id: crt.py 1470 2016-03-28 18:36:17Z mwall $
# Copyright 2013-2016 Matthew Wall
# thanks to gary roderick for significant contributions to this code

"""Emit loop data to file in Cumulus realtime format.
   http://wiki.sandaysoft.com/a/Realtime.txt

Put this file in bin/user/crt.py, then add this to your weewx.conf:

[CumulusRealTime]
    unit_system = METRIC # options are US, METRIC, METRICWX
    date_separator = /
    filename = /path/to/realtime.txt
    none = NULL

[Engine]
    [[Services]]
        archive_services = ..., user.crt.CumulusRealTime

If no unit_system is specified, the units will be those of the database.

Note that most of the code in this file is to calculate/lookup data that
are not directly provided by weewx in a LOOP packet.

The cumulus 'specification' for realtime.txt is ambiguous in places:

  - pressure trend interval is not specified, we use 3 hours for pressure
  - temperature trend interval is not specified, we use 3 hours for temperature
  
The following assumptions have been made based on the equivalent Cumulus 
Webtags listed at http://wiki.sandaysoft.com/a/Realtime.txt and 
  http://wiki.sandaysoft.com/a/Webtags:
  - wind avg speed/wind avg dir interval is not specified. The equivalent 
    Cumulus Webtags for wind avg speed and avg wind direction default to 
    10 minute averages so we use 10 minutes as well.
  - wind direction bearings in degrees are set from > 0 to 360 with 0
    indicating calm (refer to the #avgbearing Webtag at
       http://wiki.sandaysoft.com/a/Webtags) 
    (this is different to the standard used in weewx)
  
The following assumptions have been made based on examination of realtime.txt
instances from a number of live Cumulus sites:
  - time zone is not specified, local time is used throughout
  - how to handle None/NULL is not specified.  Examination of realtime.txt 
    from a number of live Cumulus sites indicates when there is no wind 
    (average or gust) wind speeds are set to zero. For other fields that
    may be None/Null we return the 'none' parameter setting (default = NULL) 
    from the weewx.conf [CumulusRealTime] section.
  - ordinal wind directions are set to --- when there is no wind.
"""

import math
import time
import syslog
from distutils.version import StrictVersion

import weewx
import weewx.almanac
import weewx.manager
import weewx.wxformulas
import weeutil.weeutil
import weedb
from weewx.engine import StdService

VERSION = "0.17"

REQUIRED_WEEWX = "3.2.0"
if StrictVersion(weewx.__version__) < StrictVersion(REQUIRED_WEEWX):
    raise weewx.UnsupportedFeature("weewx %s or greater is required, found %s"
                                   % (REQUIRED_WEEWX, weewx.__version__))

def logmsg(level, msg):
    syslog.syslog(level, 'crt: %s' % msg)

def logdbg(msg):
    logmsg(syslog.LOG_DEBUG, msg)

def loginf(msg):
    logmsg(syslog.LOG_INFO, msg)

def logerr(msg):
    logmsg(syslog.LOG_ERR, msg)

COMPASS_POINTS = ['N', 'NNE', 'NE', 'ENE', 'E', 'ESE', 'SE', 'SSE',
                  'S', 'SSW', 'SW', 'WSW', 'W', 'WNW', 'NW', 'NNW', 'N']

# map weewx unit names to cumulus unit names
UNITS_WIND = {'mile_per_hour': 'mph',
              'meter_per_second': 'm/s',
              'kilometer_per_hour': 'km/h',
              'km_per_hour': 'km/h',
              'knot': 'kts'}
UNITS_TEMP = {'degree_C': 'C',
              'degree_F': 'F'}
UNITS_PRES = {'inHg': 'in',
              'mbar': 'mb',
              'hPa': 'hPa'}
UNITS_RAIN = {'inch': 'in',
              'mm': 'mm'}
UNITS_ALT = {'foot': 'ft',
             'meter': 'm'}

def _convert(from_v, from_units, to_units, group):
    vt = (from_v, from_units, group)
    return weewx.units.convert(vt, to_units)[0]

def clamp_degrees(x):
    if x is not None:
        return x if x != 0.0 else 360.0
    return None

def degree_to_compass(x):
    if x is None:
        return '---'
    idx = int((x + 11.25) / 22.5)
    return COMPASS_POINTS[idx]

def get_db_units(dbm):
    val = dbm.getSql("SELECT usUnits FROM %s LIMIT 1" % dbm.table_name)
    return val[0] if val is not None else None

def calc_avg_windspeed(dbm, ts, interval=600):
    sts = ts - interval
    val = dbm.getSql("SELECT AVG(windSpeed) FROM %s "
                     "WHERE dateTime>? AND dateTime<=?" % dbm.table_name,
                     (sts, ts))
    return val[0] if val is not None else None

def calc_avg_winddir(dbm, ts, interval=600):
    sts = ts - interval
    val = dbm.getSql("SELECT AVG(windDir) FROM %s "
                     "WHERE dateTime>? AND dateTime<=?" % dbm.table_name,
                     (sts, ts))
    return clamp_degrees(val[0]) if val is not None else None

def calc_max_gust_10min(dbm, ts):
    sts = ts - 600
    val = dbm.getSql("SELECT MAX(windGust) FROM %s "
                     "WHERE dateTime>? AND dateTime<=?" % dbm.table_name,
                     (sts, ts))
    return val[0] if val is not None else None

def calc_avg_winddir_10min(dbm, ts):
    sts = ts - 600
    val = dbm.getSql("SELECT AVG(windDir) FROM %s "
                     "WHERE dateTime>? AND dateTime<=?" % dbm.table_name,
                     (sts, ts))
    return clamp_degrees(val[0]) if val is not None else None

# calculate wind run since midnight
def calc_windrun(dbm, ts, db_us):
    run = 0
    sod_ts = weeutil.weeutil.startOfDay(ts)
    for row in dbm.genSql("SELECT `interval`,windSpeed FROM %s "
                          "WHERE dateTime>? AND dateTime<=?" % dbm.table_name,
                          (sod_ts, ts)):
        if row[1] is not None:
            inc = row[1] * row[0]
            if db_us == weewx.METRICWX:
                inc *= 60.0
            else:
                inc /= 60.0
            run += inc
    return run

# get trend over past n hours, default to 3 hour window
def get_trend(label, dbm, ts, n=3):
    lastts = ts - n * 3600
    old_val = dbm.getSql("SELECT %s FROM %s "
                         "WHERE dateTime>? AND dateTime<=?" %
                         (label, dbm.table_name), (lastts, ts))
    if old_val is None or old_val[0] is None:
        return None
    return old_val[0]

def calc_trend(newval, oldval):
    if newval is None or oldval is None:
        return None
    return newval - oldval

def calc_rain_hour(dbm, ts):
    sts = ts - 3600
    val = dbm.getSql("SELECT SUM(rain) FROM %s "
                     "WHERE dateTime>? AND dateTime<=?" % dbm.table_name,
                     (sts, ts))
    return val[0] if val is not None else None

def calc_rain_month(dbm, ts):
    span = weeutil.weeutil.archiveMonthSpan(ts)
    val = dbm.getSql("SELECT SUM(rain) FROM %s "
                     "WHERE dateTime>? AND dateTime<=?" % dbm.table_name,
                     (span.start, ts))
    return val[0] if val is not None else None

def calc_rain_year(dbm, ts):
    span = weeutil.weeutil.archiveYearSpan(ts)
    val = dbm.getSql("SELECT SUM(rain) FROM %s "
                     "WHERE dateTime>? AND dateTime<=?" % dbm.table_name,
                     (span.start, ts))
    return val[0] if val is not None else None

def calc_rain_yesterday(dbm, ts):
    ts = weeutil.weeutil.startOfDay(ts)
    sts = ts - 3600 * 24
    val = dbm.getSql("SELECT SUM(rain) FROM %s "
                     "WHERE dateTime>? AND dateTime<=?" % dbm.table_name,
                     (sts, ts))
    return val[0] if val is not None else None

def calc_rain_day(dbm, ts):
    sts = weeutil.weeutil.startOfDay(ts)
    val = dbm.getSql("SELECT SUM(rain) FROM %s "
                     "WHERE dateTime>=? AND dateTime<=?" % dbm.table_name, 
                     (sts, ts))
    return val[0] if val is not None else None

def calc_ET_today(dbm, ts):
    sts = weeutil.weeutil.startOfDay(ts)
    val = dbm.getSql("SELECT SUM(ET) FROM %s "
                     "WHERE dateTime>? AND dateTime<=?" % dbm.table_name,
                     (sts, ts))
    return val[0] if val is not None else None

def calc_minmax(label, dbm, ts, minmax='MAX'):
    sts = weeutil.weeutil.startOfDay(ts)
    val = dbm.getSql("SELECT %s(%s) FROM %s "
                     "WHERE dateTime>=? AND dateTime<=?" %
                     (minmax, label, dbm.table_name), (sts, ts))
    if val is None:
        return None, None
    t = dbm.getSql("SELECT dateTime FROM %s "
                   "WHERE dateTime>=? AND dateTime<=? AND %s=?" %
                   (dbm.table_name, label), (sts, ts, val[0]))
    if t is None:
        return None, None
    tstr = time.strftime("%H:%M", time.gmtime(t[0]))
    return val[0], tstr

def calc_is_daylight(alm):
    sunrise = alm.sunrise.raw
    sunset = alm.sunset.raw
    if sunrise < alm.time_ts < sunset:
        return 1
    return 0

def calc_daylight_hours(alm):
    sunrise = alm.sunrise.raw
    sunset = alm.sunset.raw
    if alm.time_ts <= sunrise:
        return 0
    elif alm.time_ts < sunset:
        return (alm.time_ts - sunrise) / 3600.0
    return (sunset - sunrise) / 3600.0

def calc_is_sunny(rad, max_rad, threshold):
    if not rad or not max_rad:
        return 0
    if rad <= threshold * max_rad:
        return 0
    return 1

# indication of sensor contact depens on the weather station.  if the station
# has more than one indicator, then indicate failure if contact is lost with
# any one of them.
#
# Vantage
#   packet['rxCheckPercent'] == 0
#
# FineOffset
#   packet['status'] & 0x40
#
# TE923
#   packet['sensorX_state'] == STATE_MISSING_LINK
#   packet['wind_state'] == STATE_MISSING_LINK
#   packet['rain_state'] == STATE_MISSING_LINK
#   packet['uv_state'] == STATE_MISSING_LINK
#
# WMR100
# WMR200
# WMR9x8
#
# WS28xx
#   packet['rxCheckPercent'] == 0
#
# WS23xx
#   packet['cn'] == 'lost'
#
def lost_sensor_contact(packet):
    if 'rxCheckPercent' in packet and packet['rxCheckPercent'] == 0:
        return 1
    if 'cn' in packet and packet['cn'] == 'lost':
        return 1
    if (('windspeed_state' in packet and packet['windspeed_state'] == 'no_link') or
        ('rain_state' in packet and packet['rain_state'] == 'no_link') or
        ('uv_state' in packet and packet['uv_state'] == 'no_link') or
        ('h_1_state' in packet and packet['h_1_state'] == 'no_link') or
        ('h_2_state' in packet and packet['h_2_state'] == 'no_link') or
        ('h_3_state' in packet and packet['h_3_state'] == 'no_link') or
        ('h_4_state' in packet and packet['h_4_state'] == 'no_link') or
        ('h_5_state' in packet and packet['h_5_state'] == 'no_link')):
        return 1
    return 0

class ZambrettiForecast():
    DEFAULT_FORECAST_BINDING = 'forecast_binding'
    DEFAULT_BINDING_DICT = {
        'database': 'forecast_sqlite',
        'manager': 'weewx.manager.Manager',
        'table_name': 'archive',
        'schema': 'user.forecast.schema'}

    def __init__(self, config_dict):
        self.forecasting_installed = False
        self.db_max_tries = 3
        self.db_retry_wait = 3
        try:
            self.dbm_dict = weewx.manager.get_manager_dict(
                config_dict['DataBindings'],
                config_dict['Databases'],
                ZambrettiForecast.DEFAULT_FORECAST_BINDING,
                default_binding_dict=ZambrettiForecast.DEFAULT_BINDING_DICT)
            weewx.manager.open_manager(self.dbm_dict)
            self.forecasting_installed = True
        except (weedb.DatabaseError, weewx.UnsupportedFeature, KeyError):
            pass

    def is_installed(self):
        return self.forecasting_installed

    def get_zambretti_code(self):
        if not self.forecasting_installed:
            return 0

        # FIXME: add api to forecast instead of doing all the work here
        with weewx.manager.open_manager(self.dbm_dict) as dbm:
            sql = "select dateTime,zcode from %s where method = 'Zambretti' order by dateTime desc limit 1" % dbm.table_name
#            sql = "select zcode from %s where method = 'Zambretti' and dateTime = (select max(dateTime) from %s where method = 'Zambretti')" % (dbm.table_name, dbm.table_name)
            for count in range(self.db_max_tries):
                try:
                    record = dbm.getSql(sql)
                    if record is not None:
                        return ZambrettiForecast.alpha_to_number(record[1])
                except Exception, e:
                    logerr('get zambretti failed (attempt %d of %d): %s' %
                           ((count + 1), self.db_max_tries, e))
                    logdbg('waiting %d seconds before retry' %
                           self.db_retry_wait)
                    time.sleep(self.db_retry_wait)
        return 0

    # given a zambretti letter code A-Z, convert to digit 1-26
    @staticmethod
    def alpha_to_number(x):
        return ord(x) - 64


class CumulusRealTime(StdService):

    def __init__(self, engine, config_dict):
        super(CumulusRealTime, self).__init__(engine, config_dict)
        loginf("service version is %s" % VERSION)
        self.altitude_ft = weewx.units.convert(engine.stn_info.altitude_vt,
                                               "foot")[0]
        self.latitude = engine.stn_info.latitude_f
        self.longitude = engine.stn_info.longitude_f
        self.db_us = None  # unit system of the database

        d = config_dict.get('CumulusRealTime', {})
        self.filename = d.get('filename', '/var/tmp/realtime.txt')
        self.datesep = d.get('date_separator', '/')
        self.sunny_threshold = float(d.get('sunny_threshold', 0.75))
        nonesub = d.get('none', 'NULL')
        self.nonesub = nonesub if nonesub is not '' else 'NULL'
        loginf("'None' values will be displayed as %s" % self.nonesub)
        binding = d.get('binding', 'loop').lower()
        us = None
        us_label = d.get('unit_system', None)
        if us_label is not None:
            if us_label in weewx.units.unit_constants:
                loginf("units will be displayed as %s" % us_label)
                us = weewx.units.unit_constants[us_label]
            else:
                logerr("unknown unit_system %s" % us_label)
        self.unit_system = us

        self.forecast = ZambrettiForecast(config_dict)
        loginf("zambretti is installed: %s" % self.forecast.is_installed())

        if binding == 'loop':
            self.bind(weewx.NEW_LOOP_PACKET, self.handle_new_loop)
        else:
            self.bind(weewx.NEW_ARCHIVE_RECORD, self.handle_new_archive)

        loginf("binding is %s" % binding)
        loginf("output goes to %s" % self.filename)

    def handle_new_loop(self, event):
        self.handle_data(event.packet)

    def handle_new_archive(self, event):
        self.handle_data(event.record)

    def handle_data(self, event_data):
        try:
            dbm = self.engine.db_binder.get_manager('wx_binding')
            data = self.calculate(event_data, dbm)
            self.write_data(data)
        except Exception, e:
            logdbg("crt: Exception while handling data: %s" % e)
            weeutil.weeutil.log_traceback('crt: **** ')
            raise

    def write_data(self, data):
        with open(self.filename, 'w') as f:
            f.write(self.create_realtime_string(data))
            f.write("\n")

    def _cvt(self, from_v, to_us, obs, group):
        if self.db_us is None:
            return None
        from_units = weewx.units.getStandardUnitType(self.db_us, obs)[0]
        to_units = weewx.units.getStandardUnitType(to_us, obs)[0]
        return _convert(from_v, from_units, to_units, group)

    # calculate the data elements that that weewx does not provide directly.
    def calculate(self, packet, dbm):
        ts = packet.get('dateTime')
        pkt_us = packet.get('usUnits')
        if self.unit_system is not None and self.unit_system != pkt_us:
            packet = weewx.units.to_std_system(packet, self.unit_system)
            pkt_us = self.unit_system

        # the 'from' unit system is whatever the database is using.  get it
        # from the database once then cache it for use in conversions.
        if self.db_us is None:
            self.db_us = get_db_units(dbm)

        p_u = weewx.units.getStandardUnitType(pkt_us, 'barometer')[0]
        t_u = weewx.units.getStandardUnitType(pkt_us, 'outTemp')[0]
        r_u = weewx.units.getStandardUnitType(pkt_us, 'rain')[0]
        ws_u = weewx.units.getStandardUnitType(pkt_us, 'windSpeed')[0]
        alt_u = weewx.units.getStandardUnitType(pkt_us, 'altitude')[0]

        data = dict(packet)
        data['windSpeed_avg'] = self._cvt(
            calc_avg_windspeed(dbm, ts), pkt_us, 'windSpeed', 'group_speed')
        data['cumulus_windDir'] = clamp_degrees(packet.get('windDir'))
        data['windDir_compass'] = degree_to_compass(packet.get('windDir'))
        v = _convert(packet.get('windSpeed'), ws_u, 'knot', 'group_speed')
        data['windSpeed_beaufort'] = weewx.wxformulas.beaufort(v)
        data['units_wind'] = UNITS_WIND.get(ws_u, ws_u)
        data['units_temperature'] = UNITS_TEMP.get(t_u, t_u)
        data['units_pressure'] = UNITS_PRES.get(p_u, p_u)
        data['units_rain'] = UNITS_RAIN.get(r_u, r_u)
        wr = calc_windrun(dbm, ts, self.db_us)
        data['windrun'] = self._cvt(wr, pkt_us, 'windrun', 'group_distance')
        p1 = packet.get('barometer')
        p2 = get_trend('barometer', dbm, ts)
        p2 = self._cvt(p2, pkt_us, 'barometer', 'group_pressure')
        data['pressure_trend'] = calc_trend(p1, p2)
        t1 = packet.get('outTemp')
        t2 = get_trend('outTemp', dbm, ts)
        t2 = self._cvt(t2, pkt_us, 'outTemp', 'group_temperature')
        data['temperature_trend'] = calc_trend(t1, t2)
        data['rain_month'] = self._cvt(
            calc_rain_month(dbm, ts), pkt_us, 'rain', 'group_rain')
        data['rain_year'] = self._cvt(
            calc_rain_year(dbm, ts), pkt_us, 'rain', 'group_rain')
        data['rain_yesterday'] = self._cvt(
            calc_rain_yesterday(dbm, ts), pkt_us, 'rain', 'group_rain')
        data['dayRain'] = self._cvt(
            calc_rain_day(dbm, ts), pkt_us, 'rain', 'group_rain')
        v, t = calc_minmax('outTemp', dbm, ts, 'MAX')
        data['outTemp_max'] = self._cvt(
            v, pkt_us, 'outTemp', 'group_temperature')
        data['outTemp_max_time'] = t
        v, t = calc_minmax('outTemp', dbm, ts, 'MIN')
        data['outTemp_min'] = self._cvt(
            v, pkt_us, 'outTemp', 'group_temperature')
        data['outTemp_min_time'] = t
        v, t = calc_minmax('windSpeed', dbm, ts, 'MAX')
        data['windSpeed_max'] = self._cvt(
            v, pkt_us, 'windSpeed', 'group_speed')
        data['windSpeed_max_time'] = t
        v, t = calc_minmax('windGust', dbm, ts, 'MAX')
        data['windGust_max'] = self._cvt(
            v, pkt_us, 'windGust', 'group_speed')
        data['windGust_max_time'] = t
        v, t = calc_minmax('barometer', dbm, ts, 'MAX')
        data['pressure_max'] = self._cvt(
            v, pkt_us, 'barometer', 'group_pressure')
        data['pressure_max_time'] = t
        v, t = calc_minmax('barometer', dbm, ts, 'MIN')
        data['pressure_min'] = self._cvt(
            v, pkt_us, 'barometer', 'group_pressure')
        data['pressure_min_time'] = t
        data['10min_high_gust'] = self._cvt(
            calc_max_gust_10min(dbm, ts), pkt_us, 'windSpeed', 'group_speed')

        data['ET_today'] = calc_ET_today(dbm, ts)
        data['10min_avg_wind_bearing'] = calc_avg_winddir_10min(dbm, ts)
        data['rain_hour'] = self._cvt(
            calc_rain_hour(dbm, ts), pkt_us, 'rain', 'group_rain')
        data['lost_sensors_contact'] = lost_sensor_contact(packet)
        data['avg_wind_dir'] = degree_to_compass(data['10min_avg_wind_bearing'])
        data['units_cloudbase'] = UNITS_ALT.get(alt_u)
        t_C = _convert(packet.get('outTemp'),
                       t_u, 'degree_C', 'group_temperature')
        alt_m = _convert(self.altitude_ft, 'foot', 'meter', 'group_altitude')
        p_mbar = _convert(packet.get('barometer'), p_u, 
                          'mbar', 'group_pressure')
        alm = weewx.almanac.Almanac(ts, self.latitude, self.longitude, alt_m,
                                    t_C, p_mbar)
        data['is_daylight'] = calc_is_daylight(alm)
        data['sunshine_hours'] = calc_daylight_hours(alm)
        data['is_sunny'] = data['is_daylight']
        data['zambretti_code'] = self.forecast.get_zambretti_code()
        return data

    def format(self, data, label, places=None):
        value = data.get(label)
        if value is None:
            value = self.nonesub
        elif places is not None:
            try:
                v = float(value)
                fmt = "%%.%df" % places
                value = fmt % v
            except ValueError:
                pass
        return str(value)

    # the * indicates a field that is not part of a typical LOOP packet
    # the ## indicates calculation is not yet implemented
    def create_realtime_string(self, data):
        fields = []
        datefmt = "%%d%s%%m%s%%y" % (self.datesep, self.datesep)
        tstr = time.strftime(datefmt, time.localtime(data['dateTime']))
        fields.append(tstr)                                           # 1
        tstr = time.strftime("%H:%M:%S", time.localtime(data['dateTime']))
        fields.append(tstr)                                           # 2
        fields.append(self.format(data, 'outTemp', 1))                # 3
        fields.append(self.format(data, 'outHumidity', 0))            # 4
        fields.append(self.format(data, 'dewpoint', 1))               # 5
        fields.append(self.format(data, 'windSpeed_avg', 1))          # 6  *
        fields.append(self.format(data, 'windSpeed', 1))              # 7
        fields.append(self.format(data, 'cumulus_windDir', 0))        # 8
        fields.append(self.format(data, 'rainRate', 1))               # 9
        fields.append(self.format(data, 'dayRain', 1))                # 10
        fields.append(self.format(data, 'barometer', 1))              # 11
        fields.append(self.format(data, 'windDir_compass'))           # 12 *
        fields.append(self.format(data, 'windSpeed_beaufort'))        # 13 *
        fields.append(self.format(data, 'units_wind'))                # 14 *
        fields.append(self.format(data, 'units_temperature'))         # 15 *
        fields.append(self.format(data, 'units_pressure'))            # 16 *
        fields.append(self.format(data, 'units_rain'))                # 17 *
        fields.append(self.format(data, 'windrun', 1))               # 18 *
        fields.append(self.format(data, 'pressure_trend', 1))         # 19 *
        fields.append(self.format(data, 'rain_month', 1))             # 20 *
        fields.append(self.format(data, 'rain_year', 1))              # 21 *
        fields.append(self.format(data, 'rain_yesterday', 1))         # 22 *
        fields.append(self.format(data, 'inTemp', 1))                 # 23
        fields.append(self.format(data, 'inHumidity', 0))             # 24
        fields.append(self.format(data, 'windchill', 1))              # 25
        fields.append(self.format(data, 'temperature_trend', 1))      # 26 *
        fields.append(self.format(data, 'outTemp_max', 1))            # 27 *
        fields.append(self.format(data, 'outTemp_max_time'))          # 28 *
        fields.append(self.format(data, 'outTemp_min', 1))            # 29 *
        fields.append(self.format(data, 'outTemp_min_time'))          # 30 *
        fields.append(self.format(data, 'windSpeed_max', 1))          # 31 *
        fields.append(self.format(data, 'windSpeed_max_time'))        # 32 *
        fields.append(self.format(data, 'windGust_max', 1))           # 33 *
        fields.append(self.format(data, 'windGust_max_time'))         # 34 *
        fields.append(self.format(data, 'pressure_max', 1))           # 35 *
        fields.append(self.format(data, 'pressure_max_time'))         # 36 *
        fields.append(self.format(data, 'pressure_min', 1))           # 37 *
        fields.append(self.format(data, 'pressure_min_time'))         # 38 *
        fields.append('%s' % weewx.__version__)                       # 39
        fields.append('0')                                            # 40
        fields.append(self.format(data, '10min_high_gust', 1))        # 41 *
        fields.append(self.format(data, 'heatindex', 1))              # 42 *
        fields.append(self.format(data, 'humidex', 1))                # 43 *
        fields.append(self.format(data, 'UV', 0))                     # 44
        fields.append(self.format(data, 'ET_today', 1))               # 45 *
        fields.append(self.format(data, 'radiation', 0))              # 46
        fields.append(self.format(data, '10min_avg_wind_bearing', 0)) # 47 *
        fields.append(self.format(data, 'rain_hour'))                 # 48 *
        fields.append(self.format(data, 'zambretti_code'))            # 49 *
        fields.append(self.format(data, 'is_daylight'))               # 50 *
        fields.append(self.format(data, 'lost_sensors_contact'))      # 51 *
        fields.append(self.format(data, 'avg_wind_dir'))              # 52 *
        fields.append(self.format(data, 'cloudbase', 0))              # 53 *
        fields.append(self.format(data, 'units_cloudbase'))           # 54 *
        fields.append(self.format(data, 'appTemp', 1))                # 55 *
        fields.append(self.format(data, 'sunshine_hours', 1))         # 56 *
        fields.append(self.format(data, 'maxSolarRad', 1))            # 57
        fields.append(self.format(data, 'is_sunny'))                  # 58 *
        return ' '.join(fields)
