# $Id: crt.py 1347 2015-08-02 14:48:46Z tkeffer $
# Copyright 2013 Matthew Wall
"""Emit loop data to file in Cumulus realtime format.
   http://wiki.sandaysoft.com/a/Realtime.txt

Put this file in bin/user/crt.py, then add this to your weewx.conf:

[CumulusRealTime]
    unit_system = METRIC # options are US, METRIC, METRICWX
    date_separator = /
    filename = /path/to/realtime.txt

[Engine]
    [[Services]]
        archive_services = ..., user.crt.CumulusRealTime

If no unit_system is specified, the units will be those of the database.

Note that most of the code in this file is to calculate/lookup data that
are not directly provided by weewx in a LOOP packet.

The cumulus 'specification' for realtime.txt is ambiguous:

  pressure trend interval is not specified, we use 3 hours for pressure
  temperature trend interval is not specified, we use 3 hours for temperature
  wind avg speed/dir interval is not specified, we use 10 minutes
  time zone is not specified, we use utc everywhere
  pressure is not specified, we use 'barometer' not 'pressure' or 'altimeter'

  how to handle None/NULL is not specified.  we return a NULL when the value
  is None, but the cumulus specification does not indicate a correct
  behavior.
"""

# FIXME: is_sunny is ill-defined.  what radiation threshold indicates true?
# FIXME: is rainYear past 365 days or since 1 jan?
# FIXME: is rainMonth past 30 days or since 1st of month?

import math
import time
import syslog

import weewx
import weewx.almanac
import weewx.wxformulas
import weeutil.weeutil
import weeutil.Sun
from weewx.engine import StdService

VERSION = "0.13"

if weewx.__version__ < "3":
    raise weewx.UnsupportedFeature("weewx 3 is required, found %s" %
                                   weewx.__version__)

def logmsg(level, msg):
    syslog.syslog(level, 'crt: %s' % msg)

def logdbg(msg):
    logmsg(syslog.LOG_DEBUG, msg)

def loginf(msg):
    logmsg(syslog.LOG_INFO, msg)

def logerr(msg):
    logmsg(syslog.LOG_ERR, msg)

KM_PER_MILE = 1.60934

COMPASS_POINTS = ['N','NNE','NE','ENE','E','ESE','SE','SSE',
                  'S','SSW','SW','WSW','W','WNW','NW','NNW','N']

# map weewx unit names to cumulus unit names
UNITS_WIND = {'mile_per_hour': 'mph',
              'meter_per_second': 'm/s',
              'kilometer_per_hour': 'km/h',
              'km_per_hour': 'km/h',
              'knot': 'kts'}
UNITS_TEMP = {'degree_C': 'C',
              'degree_F': 'F'}
UNITS_PRES = {'inHg': 'in',
              'mbar': 'mb',
              'hPa': 'hPa'}
UNITS_RAIN = {'in': 'in',
              'mm': 'mm'}

def _convert(from_v, from_units, to_units, group):
    vt = (from_v, from_units, group)
    return weewx.units.convert(vt, to_units)[0]

def degreeToCompass(x):
    if x is None:
        return None
    idx = int((x + 11.25) / 22.5)
    return COMPASS_POINTS[idx]

# http://www.spc.noaa.gov/faq/tornado/beaufort.html
def knotToBeaufort(x):
    if x is None:
        return None
    if x < 1:
        return 0  # calm
    elif x < 4:
        return 1  # light air
    elif x < 7:
        return 2  # light breeze
    elif x < 11:
        return 3  # gentle breeze
    elif x < 17:
        return 4  # moderate breeze
    elif x < 22:
        return 5  # fresh breeze
    elif x < 28:
        return 6  # strong breeze
    elif x < 34:
        return 7  # near gale
    elif x < 41:
        return 8  # gale
    elif x < 48:
        return 9  # strong gale
    elif x < 56:
        return 10 # storm
    elif x < 64:
        return 11 # violent storm
    return 12 # hurricane

def get_db_units(dbm):
    val = dbm.getSql("SELECT usUnits FROM %s LIMIT 1" % dbm.table_name)
    return val[0] if val is not None else None

def calcAvgWindSpeed(dbm, ts, interval=600):
    sts = ts - interval
    val = dbm.getSql("SELECT AVG(windSpeed) FROM %s "
                     "WHERE dateTime>? AND dateTime<=?" % dbm.table_name,
                     (sts, ts))
    return val[0] if val is not None else None

def calcAvgWindDir(dbm, ts, interval=600):
    sts = ts - interval
    val = dbm.getSql("SELECT AVG(windDir) FROM %s "
                     "WHERE dateTime>? AND dateTime<=?" % dbm.table_name,
                     (sts, ts))
    return val[0] if val is not None else None

def calc10MinHighGust(dbm, ts):
    sts = ts - 600
    val = dbm.getSql("SELECT MAX(windGust) FROM %s "
                     "WHERE dateTime>? AND dateTime<=?" % dbm.table_name,
                     (sts, ts))
    return val[0] if val is not None else None

def calc10MinAvgWindDir(dbm, ts):
    sts = ts - 600
    val = dbm.getSql("SELECT AVG(windDir) FROM %s "
                     "WHERE dateTime>? AND dateTime<=?" % dbm.table_name,
                     (sts, ts))
    return val[0] if val is not None else None

# calculate wind run since midnight
def calcWindRun(dbm, ts, db_us):
    run = 0
    sod_ts = weeutil.weeutil.startOfDay(ts)
    for row in dbm.genSql("SELECT `interval`,windSpeed FROM %s "
                          "WHERE dateTime>? AND dateTime<=?" % dbm.table_name,
                          (sod_ts, ts)):
        if row[1] is not None:
            inc = row[1] * row[0]
            if db_us == weewx.METRICWX:
                inc *= 60.0
            else:
                inc /= 60.0
            run += inc
    return run

# get trend over past n hours, default to 3 hour window
def getTrend(label, dbm, ts, n=3):
    lastts = ts - n * 3600
    old_val = dbm.getSql("SELECT %s FROM %s "
                         "WHERE dateTime>? AND dateTime<=?" %
                         (label, dbm.table_name), (lastts, ts))
    if old_val is None or old_val[0] is None:
        return None
    return old_val[0]

def calcTrend(newval, oldval):
    if newval is None or oldval is None:
        return None
    return newval - oldval

def calcRainHour(dbm, ts):
    sts = ts - 3600
    val = dbm.getSql("SELECT SUM(rain) FROM %s "
                     "WHERE dateTime>? AND dateTime<=?" % dbm.table_name,
                     (sts, ts))
    return val[0] if val is not None else None

def calcRainMonth(dbm, ts):
    span = weeutil.weeutil.archiveMonthSpan(ts)
    val = dbm.getSql("SELECT SUM(rain) FROM %s "
                         "WHERE dateTime>? AND dateTime<=?" % dbm.table_name,
                         (span.start, ts))
    return val[0] if val is not None else None

def calcRainYear(dbm, ts):
    span = weeutil.weeutil.archiveYearSpan(ts)
    val = dbm.getSql("SELECT SUM(rain) FROM %s "
                     "WHERE dateTime>? AND dateTime<=?" % dbm.table_name,
                     (span.start, ts))
    return val[0] if val is not None else None

def calcRainYesterday(dbm, ts):
    ts = weeutil.weeutil.startOfDay(ts)
    sts = ts - 3600*24;
    val = dbm.getSql("SELECT SUM(rain) FROM %s "
                     "WHERE dateTime>? AND dateTime<=?" % dbm.table_name,
                     (sts, ts))
    return val[0] if val is not None else None

def calcDayRain(dbm, ts):
    sts = weeutil.weeutil.startOfDay(ts)
    val = dbm.getSql("SELECT SUM(rain) FROM %s "
                     "WHERE dateTime>=? AND dateTime<=?" % dbm.table_name, 
                     (sts, ts))
    return val[0] if val is not None else None

def calcETToday(dbm, ts):
    sts = weeutil.weeutil.startOfDay(ts)
    val = dbm.getSql("SELECT SUM(ET) FROM %s "
                     "WHERE dateTime>? AND dateTime<=?" % dbm.table_name,
                     (sts, ts))
    return val[0] if val is not None else None

def calcMinMax(label, dbm, ts, minmax='MAX'):
    sts = weeutil.weeutil.startOfDay(ts)
    val = dbm.getSql("SELECT %s(%s) FROM %s "
                     "WHERE dateTime>=? AND dateTime<=?" %
                     (minmax, label, dbm.table_name), (sts, ts))
    if val is None:
        return None, None
    t = dbm.getSql("SELECT dateTime FROM %s "
                   "WHERE dateTime>=? AND dateTime<=? AND %s=?" %
                   (dbm.table_name, label), (sts, ts, val[0]))
    if t is None:
        return None, None
    tstr = time.strftime("%H:%M", time.gmtime(t[0]))
    return val[0],tstr

def calcHumidex(t_C, dewpoint_K):
    if t_C is None or dewpoint_K is None:
        return None
    v = 1/273.16
    v -= 1/dewpoint_K
    v *= 5417.7530
    v = math.exp(v)
    v *= 6.11
    v -= 10
    v *= 0.5555
    return t_C + v

def calcIsDaylight(ts_utc, lat, lon):
    x = weeutil.weeutil.startOfDayUTC(ts_utc)
    x_tt = time.gmtime(x)
    y, m, d = x_tt[:3]
    (sunrise_utc, sunset_utc) = weeutil.Sun.sunRiseSet(y, m, d, lon, lat)
    if sunrise_utc < ts_utc < sunset_utc:
        return 1
    return 0

def calcDaylightHours(ts_utc, lat, lon):
    x = weeutil.weeutil.startOfDayUTC(ts_utc)
    x_tt = time.gmtime(x)
    y, m, d = x_tt[:3]
    (sunrise_utc, sunset_utc) = weeutil.Sun.sunRiseSet(y, m, d, lon, lat)
    if ts_utc <= sunrise_utc:
        return 0
    elif ts_utc < sunset_utc:
        return (ts_utc - sunrise_utc) / 60
    return (sunset_utc - sunrise_utc) / 60

def calcCloudBase(outTemp_F, dewpoint_F, altitude_foot):
    if outTemp_F is None or dewpoint_F is None:
        return None
    return int((outTemp_F - dewpoint_F) / 4.4 * 1000) + altitude_foot

def calcMaxRad(alm, altitude_m):
    try:
        rs_el = alm.sun.alt
        rs_R = alm.sun.earth_distance
        rs_atc = 0.8
        rs_z = altitude_m
        rs_r0 = 1367
        sinal = math.sin(rs_el)
        if sinal < 0:
            return 0
        al = math.asin(sinal)
        a0 = al * 57.2957795
        rm = math.pow((288-0.0065*rs_z)/288,5.256)/(sinal+0.15*math.pow(a0+3.885,-1.253))
        rs_toa = rs_r0 * sinal / (rs_R * rs_R)
        return rs_toa * math.pow(rs_atc, rm)
    except ValueError, e:
        logdbg("cannot calculate max solar radiation: %s" % e)
        return 0

# this depends on the weather station.  many stations have more than one.
#
# Vantage
#   packet['rxCheckPercent'] == 0
#
# FineOffset
#   packet['status'] & 0x40
#
# TE923
#   packet['sensorX_state'] == STATE_MISSING_LINK
#   packet['wind_state'] == STATE_MISSING_LINK
#   packet['rain_state'] == STATE_MISSING_LINK
#   packet['uv_state'] == STATE_MISSING_LINK
#
# WMR100
# WMR200
# WMR9x8
#
# WS28xx
#   packet['rxCheckPercent'] == 0
#
# WS23xx
#   packet['cn'] == 'lost'
#
def lostSensorContact(packet):
    if 'rxCheckPercent' in packet and packet['rxCheckPercent'] == 0:
        return 1
    if 'cn' in packet and packet['cn'] == 'lost':
        return 1
    if (('windspeed_state' in packet and packet['windspeed_state'] == 'no_link') or
        ('rain_state' in packet and packet['rain_state'] == 'no_link') or
        ('uv_state' in packet and packet['uv_state'] == 'no_link') or
        ('h_1_state' in packet and packet['h_1_state'] == 'no_link') or
        ('h_2_state' in packet and packet['h_2_state'] == 'no_link') or
        ('h_3_state' in packet and packet['h_3_state'] == 'no_link') or
        ('h_4_state' in packet and packet['h_4_state'] == 'no_link') or
        ('h_5_state' in packet and packet['h_5_state'] == 'no_link')):
        return 1
    return 0


class CumulusRealTime(StdService):

    def __init__(self, engine, config_dict):
        super(CumulusRealTime, self).__init__(engine, config_dict)
        loginf("service version is %s" % VERSION)
        self.altitude_ft = weewx.units.convert(engine.stn_info.altitude_vt,
                                               "foot")[0]
        self.latitude = engine.stn_info.latitude_f
        self.longitude = engine.stn_info.longitude_f
        self.db_us = None  # unit system of the database

        d = config_dict.get('CumulusRealTime', {})
        self.filename = d.get('filename', '/var/tmp/realtime.txt')
        self.datesep = d.get('date_separator', '/')
        binding = d.get('binding', 'loop').lower()
        us = None
        us_label = d.get('unit_system', None)
        if us_label is not None:
            if us_label in weewx.units.unit_constants:
                loginf("units will be displayed as %s" % us_label)
                us = weewx.units.unit_constants[us_label]
            else:
                logerr("unknown unit_system %s" % us_label)
        self.unit_system = us

        if binding == 'loop':
            self.bind(weewx.NEW_LOOP_PACKET, self.handle_new_loop)
        else:
            self.bind(weewx.NEW_ARCHIVE_RECORD, self.handle_new_archive)

        loginf("binding is %s" % binding)
        loginf("output goes to %s" % self.filename)

    def handle_new_loop(self, event):
        self.handle_data(event.packet)

    def handle_new_archive(self, event):
        self.handle_data(event.record)

    def handle_data(self, event_data):
        try:
            dbm = self.engine.db_binder.get_manager('wx_binding')
            data = self.calculate(event_data, dbm)
            self.write_data(data)
        except Exception, e:
            logdbg("crt: Exception while handling data: %s" % e)
            weeutil.weeutil.log_traceback('crt: **** ')
            raise

    def write_data(self, data):
        with open(self.filename, 'w') as f:
            f.write(self.create_realtime_string(data))
            f.write("\n")

    def _cvt(self, from_v, to_us, obs, group):
        from_units = weewx.units.getStandardUnitType(self.db_us, obs)[0]
        to_units = weewx.units.getStandardUnitType(to_us, obs)[0]
        return _convert(from_v, from_units, to_units, group)

    # calculate the data elements that that weewx does not provide directly.
    def calculate(self, packet, dbm):
        ts = packet.get('dateTime')
        pkt_us = packet.get('usUnits')
        if self.unit_system is not None and self.unit_system != pkt_us:
            packet = weewx.units.to_std_system(packet, self.unit_system)
            pkt_us = self.unit_system

        # the 'from' unit system is whatever the database is using.  get it
        # from the database once then cache it for use in conversions.
        if self.db_us is None:
            self.db_us = get_db_units(dbm)

        p_u = weewx.units.getStandardUnitType(pkt_us, 'barometer')[0]
        t_u = weewx.units.getStandardUnitType(pkt_us, 'outTemp')[0]
        r_u = weewx.units.getStandardUnitType(pkt_us, 'rain')[0]
        ws_u = weewx.units.getStandardUnitType(pkt_us, 'windSpeed')[0]

        data = dict(packet)
        data['windSpeed_avg'] = self._cvt(
            calcAvgWindSpeed(dbm, ts), pkt_us, 'windSpeed', 'group_speed')
        data['windDir_compass'] = degreeToCompass(packet.get('windDir'))
        v = _convert(packet.get('windSpeed'), ws_u, 'knot', 'group_speed')
        data['windSpeed_beaufort'] = knotToBeaufort(v)
        data['units_wind'] = UNITS_WIND.get(ws_u, ws_u)
        data['units_temperature'] = UNITS_TEMP.get(t_u, t_u)
        data['units_pressure'] = UNITS_PRES.get(p_u, p_u)
        data['units_rain'] = UNITS_RAIN.get(r_u, r_u)
        # weewx does not know distance, so we have to convert this directly
        wr = calcWindRun(dbm, ts, self.db_us)
        if self.db_us != pkt_us:
            if self.db_us == weewx.US:
                if pkt_us == weewx.METRIC:
                    wr *= KM_PER_MILE
                elif pkt_us == weewx.METRICWX:
                    wr *= KM_PER_MILE * 1000
            elif self.db_us == weewx.METRIC:
                if pkt_us == weewx.US:
                    wr /= KM_PER_MILE
                elif pkt_us == weewx.METRICWX:
                    wr *= 1000
            elif self.db_us == weewx.METRICWX:
                if pkt_us == weewx.US:
                    wr /= (KM_PER_MILE * 1000)
                elif pkt_us == weewx.METRIC:
                    wr /= 1000
        # wind run is mile, kilometer, or meter
        data['wind_run'] = wr
        p1 = packet.get('barometer')
        p2 = getTrend('barometer', dbm, ts)
        p2 = self._cvt(p2, pkt_us, 'barometer', 'group_pressure')
        data['pressure_trend'] = calcTrend(p1, p2)
        t1 = packet.get('outTemp')
        t2 = getTrend('outTemp', dbm, ts)
        t2 = self._cvt(t2, pkt_us, 'outTemp', 'group_temperature')
        data['temperature_trend'] = calcTrend(t1, t2)
        data['rain_month'] = self._cvt(
            calcRainMonth(dbm, ts), pkt_us, 'rain', 'group_rain')
        data['rain_year'] = self._cvt(
            calcRainYear(dbm, ts), pkt_us, 'rain', 'group_rain')
        data['rain_yesterday'] = self._cvt(
            calcRainYesterday(dbm, ts), pkt_us, 'rain', 'group_rain')
        data['dayRain'] = self._cvt(
            calcDayRain(dbm, ts), pkt_us, 'rain', 'group_rain')
        v,t = calcMinMax('outTemp', dbm, ts, 'MAX')
        data['outTemp_max'] = self._cvt(
            v, pkt_us, 'outTemp', 'group_temperature')
        data['outTemp_max_time'] = t
        v,t = calcMinMax('outTemp', dbm, ts, 'MIN')
        data['outTemp_min'] = self._cvt(
            v, pkt_us, 'outTemp', 'group_temperature')
        data['outTemp_min_time'] = t
        v,t = calcMinMax('windSpeed', dbm, ts, 'MAX')
        data['windSpeed_max'] = self._cvt(
            v, pkt_us, 'windSpeed', 'group_speed')
        data['windSpeed_max_time'] = t
        v,t = calcMinMax('windGust', dbm, ts, 'MAX')
        data['windGust_max'] = self._cvt(
            v, pkt_us, 'windGust', 'group_speed')
        data['windGust_max_time'] = t
        v,t = calcMinMax('barometer', dbm, ts, 'MAX')
        data['pressure_max'] = self._cvt(
            v, pkt_us, 'barometer', 'group_pressure')
        data['pressure_max_time'] = t
        v,t = calcMinMax('barometer', dbm, ts, 'MIN')
        data['pressure_min'] = self._cvt(
            v, pkt_us, 'barometer', 'group_pressure')
        data['pressure_min_time'] = t
        data['10min_high_gust'] = self._cvt(
            calc10MinHighGust(dbm, ts), pkt_us, 'windSpeed', 'group_speed')

        t_F = _convert(packet.get('outTemp'),
                       t_u, 'degree_F', 'group_temperature')
        x = weewx.wxformulas.heatindexF(t_F, packet.get('outHumidity'))
        data['heatindex'] = _convert(x, 'degree_F', t_u, 'group_temperature')
        t_C = _convert(packet.get('outTemp'),
                       t_u, 'degree_C', 'group_temperature')
        dp_C = _convert(packet.get('dewpoint'),
                        t_u, 'degree_C', 'group_temperature')
        dp_K = dp_C + 273.15 if dp_C is not None else None
        v = calcHumidex(t_C, dp_K);
        data['humidex'] = _convert(v, 'degree_C', t_u, 'group_temperature')
        data['ET_today'] = calcETToday(dbm, ts)
        data['10min_avg_wind_bearing'] = calc10MinAvgWindDir(dbm, ts)
        data['rain_hour'] = self._cvt(
            calcRainHour(dbm, ts), pkt_us, 'rain', 'group_rain')
        data['is_daylight'] = calcIsDaylight(ts, self.latitude, self.longitude)
        data['lost_sensors_contact'] = lostSensorContact(packet)
        data['avg_wind_dir'] = degreeToCompass(calcAvgWindDir(dbm, ts))
        dp_F = _convert(packet.get('dewpoint'),
                        t_u, 'degree_F', 'group_temperature')
        cb_f = calcCloudBase(t_F, dp_F, self.altitude_ft)
        ut = weewx.units.getStandardUnitType(pkt_us, 'altitude')
        vt = (cb_f, "foot", "group_altitude")
        data['cloud_base'] = weewx.units.convert(vt, ut[0])[0]
        data['units_cloud_base'] = ut[0]
        if (v < 50):
            ws_mph = _convert(packet.get('windSpeed'),
                              ws_u, 'mile_per_hour', 'group_speed')
            x = weewx.wxformulas.windchillF(t_F, ws_mph)
            v = _convert(x, 'degree_F', t_u, 'group_temperature')
        else:
            v = data['heatindex']
        data['apparent_temperature'] = v
        data['sunshine_hours'] = calcDaylightHours(
            ts, self.latitude, self.longitude)
        alt_m = self.altitude_ft * 0.3048
        p_mbar = _convert(packet.get('barometer'), p_u, 'mbar', 'group_pressure')
        alm = weewx.almanac.Almanac(ts, self.latitude, self.longitude, alt_m,
                                    t_C, p_mbar)
        data['max_rad'] = calcMaxRad(alm, alt_m)
        data['is_sunny'] = data['is_daylight']
        return data

    def format(self, data, label, places=None):
        value = data.get(label)
        if value is None:
            value = 'NULL'
        elif places is not None:
            try:
                v = float(value)
                fmt = "%%.%df" % places
                value = fmt % v
            except ValueError:
                pass
        return str(value)

    # the * indicates a field that is not part of a typical LOOP packet
    # the ## indicates calculation is not yet implemented
    def create_realtime_string(self, data):
        fields = []
        datefmt = "%%d%s%%m%s%%y" % (self.datesep, self.datesep)
        tstr = time.strftime(datefmt, time.gmtime(data['dateTime']))
        fields.append(tstr)                                           # 1
        tstr = time.strftime("%H:%M:%S", time.gmtime(data['dateTime']))
        fields.append(tstr)                                           # 2
        fields.append(self.format(data, 'outTemp', 1))                # 3
        fields.append(self.format(data, 'outHumidity', 0))            # 4
        fields.append(self.format(data, 'dewpoint', 1))               # 5
        fields.append(self.format(data, 'windSpeed_avg', 1))          # 6  *
        fields.append(self.format(data, 'windSpeed', 1))              # 7
        fields.append(self.format(data, 'windDir', 0))                # 8
        fields.append(self.format(data, 'rainRate', 1))               # 9
        fields.append(self.format(data, 'dayRain', 1))                # 10
        fields.append(self.format(data, 'barometer', 1))              # 11
        fields.append(self.format(data, 'windDir_compass'))           # 12 *
        fields.append(self.format(data, 'windSpeed_beaufort'))        # 13 *
        fields.append(self.format(data, 'units_wind'))                # 14 *
        fields.append(self.format(data, 'units_temperature'))         # 15 *
        fields.append(self.format(data, 'units_pressure'))            # 16 *
        fields.append(self.format(data, 'units_rain'))                # 17 *
        fields.append(self.format(data, 'wind_run', 1))               # 18 *
        fields.append(self.format(data, 'pressure_trend', 1))         # 19 *
        fields.append(self.format(data, 'rain_month', 1))             # 20 *
        fields.append(self.format(data, 'rain_year', 1))              # 21 *
        fields.append(self.format(data, 'rain_yesterday', 1))         # 22 *
        fields.append(self.format(data, 'inTemp', 1))                 # 23
        fields.append(self.format(data, 'inHumidity', 0))             # 24
        fields.append(self.format(data, 'windchill', 1))              # 25
        fields.append(self.format(data, 'temperature_trend', 1))      # 26 *
        fields.append(self.format(data, 'outTemp_max', 1))            # 27 *
        fields.append(self.format(data, 'outTemp_max_time'))          # 28 *
        fields.append(self.format(data, 'outTemp_min', 1))            # 29 *
        fields.append(self.format(data, 'outTemp_min_time'))          # 30 *
        fields.append(self.format(data, 'windSpeed_max', 1))          # 31 *
        fields.append(self.format(data, 'windSpeed_max_time'))        # 32 *
        fields.append(self.format(data, 'windGust_max', 1))           # 33 *
        fields.append(self.format(data, 'windGust_max_time'))         # 34 *
        fields.append(self.format(data, 'pressure_max', 1))           # 35 *
        fields.append(self.format(data, 'pressure_max_time'))         # 36 *
        fields.append(self.format(data, 'pressure_min', 1))           # 37 *
        fields.append(self.format(data, 'pressure_min_time'))         # 38 *
        fields.append('%s' % weewx.__version__)                       # 39
        fields.append('0')                                            # 40
        fields.append(self.format(data, '10min_high_gust', 1))        # 41 *
        fields.append(self.format(data, 'heatindex', 1))              # 42 *
        fields.append(self.format(data, 'humidex', 1))                # 43 *
        fields.append(self.format(data, 'UV', 0))                     # 44
        fields.append(self.format(data, 'ET_today', 1))               # 45 *
        fields.append(self.format(data, 'radiation'))                 # 46
        fields.append(self.format(data, '10min_avg_wind_bearing', 0)) # 47 *
        fields.append(self.format(data, 'rain_hour'))                 # 48 *
        fields.append('0')                                            # 49
        fields.append(self.format(data, 'is_daylight'))               # 50 *
        fields.append(self.format(data, 'lost_sensors_contact'))      # 51 *
        fields.append(self.format(data, 'avg_wind_dir'))              # 52 *
        fields.append(self.format(data, 'cloud_base', 0))             # 53 *
        fields.append(self.format(data, 'units_cloud_base'))          # 54 *
        fields.append(self.format(data, 'apparent_temperature', 1))   # 55 *
        fields.append(self.format(data, 'sunshine_hours', 1))         # 56 *
        fields.append(self.format(data, 'max_rad', 1))                # 57 ##
        fields.append(self.format(data, 'is_sunny'))                  # 58 *
        return ' '.join(fields)
