# $Id$
# Copyright 2013 Matthew Wall
"""periodically 'phone home' to register a weewx station"""

# FIXME: the REST base class has stuff we do not need (e.g. archive)

import re
import syslog
import urllib
import urllib2
import time
import weewx.restful

WEEWX_SERVER_URL = 'http://weewx.com/register'

def logdbg(msg):
    syslog.syslog(syslog.LOG_DEBUG, 'register: %s' % msg)

def loginf(msg):
    syslog.syslog(syslog.LOG_INFO, 'register: %s' % msg)

def logcrt(msg):
    syslog.syslog(syslog.LOG_CRIT, 'register: %s' % msg)

def logerr(msg):
    syslog.syslog(syslog.LOG_ERR, 'register: %s' % msg)

class StationRegistry(weewx.restful.REST):
    """Class for phoning home to register a weewx station.

    This will periodically do a http GET with the following information:

      station_url           should be world-accessible
      description           description of station location
      latitude, longitude   must be in decimal format
      station_type          for example Vantage, FineOffsetUSB

    The station_url is the unique key by which a station is identified.

    The other values are obtained from the configuration in weewx.conf.

      server_url - site at which to register
      [Required]

      description: description of station location
      [Required]

      latitude: station latitude
      [Required]

      longitude: station longitude
      [Required]

      hardware: station hardware
      [Required]

      interval: time in seconds between posts
      [Optional.  Default is 604800 (once per week)]

      max_tries: number of attempts to make before giving up
      [Optional.  Default is 5]
      """

    def __init__(self, *args, **kwargs):
        self.server_url = kwargs.get('server_url', WEEWX_SERVER_URL)
        self.station_url = kwargs['station_url']
        self.description = kwargs['description']
        self.latitude = float(kwargs['latitude'])
        self.longitude = float(kwargs['longitude'])
        self.hardware = kwargs['hardware']
        self.interval = int(kwargs.get('interval', 604800))
        self.max_tries = int(kwargs.get('max_tries', 5))

        self.site = 'StationRegistry'
        self.station = self.server_url

        self._last_ts = None
        # adapted from django URLValidator
        self._urlregex = re.compile(
            r'^(?:http)s?://' # http:// or https://
            r'(?:(?:[A-Z0-9](?:[A-Z0-9-]{0,61}[A-Z0-9])?\.)+(?:[A-Z]{2,6}\.?|[A-Z0-9-]{2,}\.?)|' #domain...
            r'localhost|' #localhost...
            r'\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3})' # ...or ip
            r'(?::\d+)?' # optional port
            r'(?:/?|[/?]\S+)$', re.IGNORECASE)

        self._validateParameters()

    def postData(self, archive, time_ts):
        now = time.time()
        if self._last_ts is not None and now - self._last_ts < self.interval:
            msg = 'registration interval (%d) has not passed.' % self.interval
            logdbg(msg)
            raise weewx.restful.SkippedPost, msg

        url = self.getURL()
        for _count in range(self.max_tries):
            # Use HTTP GET to convey the station data
            try:
                logdbg("attempt to register using '%s'" % url)
                _response = urllib2.urlopen(url)
            except (urllib2.URLError, socket.error, httplib.BadStatusLine), e:
                # Unsuccessful. Log it and try again
                logerr('failed attempt %d of %d' % (_count+1, self.max_tries))
                logerr('   **** Reason: %s' % (e,))
            else:
                # Check for the server response
                for line in _response:
                    # Registration failed, log it and bail out
                    if line.startswith('FAIL'):
                        logerr("registration server returned %s" % line)
                        raise weewx.restful.FailedPost, line
                # Registration was successful
                logdbg('registration successful')
                self._last_ts = time.time()
                return
        else:
            # The upload failed max_tries times. Log it.
            msg = 'failed to register after %d tries' % self.max_tries
            logerr(msg)
            raise IOError, msg

    def getURL(self):
        url = self.server_url
        url += '?station_url=%s' % self.station_url
        url += '&description=%s' % self.description
        url += '&latitude=%f&longitude=%f' % (self.latitude, self.longitude)
        url += '&station_type=%s' % self.hardware
        return url

    def _checkURL(self, url):
        return self._urlregex.search(url)

    def _validateParameters(self):
        msgs = []
        # ensure the url does not have problem characters.  do not check
        # to see whether the site actually exists.
        if not self._checkURL(self.station_url):
            msgs.append("station_url '%s' is not a valid URL" % self.station_url)

        # check server url just in case someone modified the default
        url = self.server_url
        if not self._checkURL(url):
            msgs.append("server_url '%s' is not a valid URL" % self.server_url)

        if len(msgs) > 0:
            errmsg = 'One or more unusable station parameters.'
            logerr(errmsg)
            for m in msgs:
                logerr('   **** %s' % m)
            raise ValueError(errmsg)

        # eliminate any problematic characters from the description string
        self.description = self.description.replace('=','%3D')
        self.description = self.description.replace("'",'%60')
        self.description = self.description.replace('&','%26')
