#!/usr/bin/perl
# $Id: register.cgi 438 2013-01-31 18:51:51Z mwall $
# register/update a weewx station via GET or POST request
# Copyright 2013 Matthew Wall
#
# This CGI script takes requests from weewx stations and registers them into
# a database.  It expects the following parameters:
#
# station_url
# description (location)
# latitude
# longitude
# station_type
#
# The station_url is used to uniquely identify a station.
#
# If the station has never been seen before, a new record is added.  If the
# station has been seen, then a field is updated with the timestamp of the
# request.
#
# Data are saved to a sqlite database.  The database contains a single table
# with the following structure:
#
# create table stations (station_url varchar2(255) primary key,
#                        description varchar2(255),
#                        latitude number,
#                        longitude number,
#                        station_type varchar2(64),
#                        last_seen int)
#
# If the database does not exist, one will be created with an empty table.
#
# FIXME: should we have a field for first_seen?
# FIXME: add checks to prevent update too frequently
# FIXME: record ip address, refuse if changes too frequently

use strict;
use POSIX;

# location of the sqlite database
my $db = '/var/lib/weewx/stations.db';

# format of the date as returned in the html footers
my $DATE_FORMAT = "%Y.%m.%d %H:%M:%S UTC";

my $RMETHOD = $ENV{'REQUEST_METHOD'};
if($RMETHOD eq 'GET' || $RMETHOD eq 'POST') {
    my($qs,%rqpairs) = &getrequest;
    &handleregistration(%rqpairs);
} else {
    &writereply('Bad Request', 'FAIL', "Unsupported request method '$RMETHOD'.");
}

exit 0;



sub handleregistration {
    my(%rqpairs) = @_;

    my ($status,$msg,$rec) = registerstation(%rqpairs);
    if($status eq 'OK') {
        &writereply('Registration Complete', 'OK', $msg, $rec);
    } else {
        &writereply('Registration Failed', 'FAIL', $msg, $rec);
    }
}

# if this is a new station, add an entry to the database.  if an entry already
# exists, update the last_seen timestamp.
sub registerstation {
    my(%rqpairs) = @_;

    my %rec;
    $rec{station_url} = $rqpairs{station_url};
    $rec{description} = $rqpairs{description};
    $rec{latitude} = $rqpairs{latitude};
    $rec{longitude} = $rqpairs{longitude};
    $rec{station_type} = $rqpairs{station_type};
    my $ts = time;
    $rec{last_seen} = $ts;

    my @msgs;
    if($rec{station_url} !~ /^https?:\/\/\S+\.\S+/) {
        push @msgs, 'station_url is not a proper URL';
    }
    if($rec{station_url} =~ /'/) {
        push @msgs, 'station_url cannot contain single quotes';
    }
    if($rec{description} =~ /'/) {
        push @msgs, 'description cannot contain single quotes';
    }
    if($rec{station_type} =~ /'/) {
        push @msgs, 'station_type cannot contain single quotes';
    }
    if($rec{latitude} eq q()) {
        push @msgs, 'latitude must be specified';
    } elsif($rec{latitude} =~ /[^0-9.-]+/) {
        push @msgs, 'latitude must be decimal notation, for example 54.234 or -23.5';
    }
    if($rec{longitude} eq q()) {
        push @msgs, 'longitude must be specified';
    } elsif($rec{longitude} =~ /[^0-9.-]+/) {
        push @msgs, 'longitude must be decimal notation, for example 7.15 or -78.535';
    }
    if($#msgs >= 0) {
        my $msg = q();
        foreach my $m (@msgs) {
            $msg .= '; ' if $msg ne q();
            $msg .= $m;
        }
        return (-1, $msg, \%rec);
    }

    my $rval = eval "{ require DBI; }"; ## no critic (ProhibitStringyEval)
    if(!$rval) {
        my $msg = 'bad server configuration: DBI is not installed';
        return ('FAIL', $msg, \%rec);
    }
    my $havesqlite = 0;
    my @drivers = DBI->available_drivers();
    foreach my $d (@drivers) {
        $havesqlite = 1 if $d =~ /^sqlite/i;
    }
    if(!$havesqlite) {
        my $msg = 'bad server configuration: DBI::SQLite is not installed';
        return ('FAIL', $msg, \%rec);
    }

    my $dbexists = -f $db;
    my $dbh = DBI->connect("dbi:SQLite:$db", q(), q(), { RaiseError => 0 });
    if (!$dbh) {
        my $msg = 'connection to database failed: ' . $DBI::errstr;
        return ('FAIL', $msg, \%rec);
    }

    my $rc = 0;
    if(! $dbexists) {
        $rc = $dbh->do('create table stations(station_url varchar2(255) primary key, description varchar2(255), latitude number, longitude number, station_type varchar2(64), last_seen int)');
        if(!$rc) {
            my $msg = 'create table failed: ' . $DBI::errstr;
            $dbh->disconnect();
            return ('FAIL', $msg, \%rec);
        }
    }

    my $qs = "insert or replace into stations (station_url,description,latitude,longitude,station_type,last_seen) values ('$rec{station_url}','$rec{description}','$rec{latitude}','$rec{longitude}','$rec{station_type}',$ts)";
    $rc = $dbh->do($qs);
    if(!$rc) {
        my $msg = 'insert/replace failed: ' . $DBI::errstr;
        $dbh->disconnect();
        return ('FAIL', $msg, \%rec);
    }

    $dbh->disconnect();

    return ('OK', 'registration received', \%rec);
}

sub writereply {
    my($title, $status, $msg, $rec) = @_;

    my $tstr = &getformatteddate;
    &writecontenttype;
    &writeheader($title);
    print STDOUT "<p><strong>$title</strong></p>\n";
    print STDOUT "<pre>\n";
    print STDOUT "$status: $msg\n";
    print STDOUT "</pre>\n";
    print STDOUT "<pre>\n";
    print STDOUT "station_url: $rec->{station_url}\n";
    print STDOUT "description: $rec->{description}\n";
    print STDOUT "latitude: $rec->{latitude}\n";
    print STDOUT "longitude: $rec->{longitude}\n";
    print STDOUT "station_type: $rec->{station_type}\n";
    print STDOUT "last_seen: $rec->{last_seen}\n";
    print STDOUT "</pre>\n";
    &writefooter($tstr);
}

sub writecontenttype {
    my($type) = @_;

    $type = "text/html" if $type eq "";
    print STDOUT "Content-type: text/html\n\n";
}

sub writeheader {
    my($title) = @_;

    print STDOUT "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 3.2 Final//EN\">\n";
    print STDOUT "<html>\n";
    print STDOUT "<head>\n";
    print STDOUT "  <title>$title</title>\n";
    print STDOUT "</head>\n";
    print STDOUT "<body>\n";
};

sub writefooter {
    my($mdate) = @_;

    if($mdate) {
        print STDOUT "<small><i>\n";
        print STDOUT "$mdate\n";
        print STDOUT "</i></small>\n";
    }

    print STDOUT "\n</body>\n</html>\n";
}

sub getformatteddate {
    return strftime $DATE_FORMAT, gmtime time;
}

sub getrequest {
    my $request = q();
    if ($ENV{'REQUEST_METHOD'} eq "POST") {
        read(STDIN, $request, $ENV{'CONTENT_LENGTH'});
    } elsif ($ENV{'REQUEST_METHOD'} eq "GET" ) {
        $request = $ENV{'QUERY_STRING'};
    }
    my $delim = ',';
    my %pairs;
    foreach my $pair (split(/[&]/, $request)) {
        $pair =~ tr/+/ /;
        $pair =~ s/%(..)/pack("c",hex($1))/ge;
        my($loc) = index($pair,"=");
        my($name) = substr($pair,0,$loc);
        my($value) = substr($pair,$loc+1);
        if($pairs{$name} eq "") {
            $pairs{$name} = $value;
        } else {
            $pairs{$name} .= "${delim}$value";
        }
    }
    return($ENV{'QUERY_STRING'},%pairs);
}
