#!/usr/bin/perl
# $Id: mksites.pl 442 2013-02-01 22:34:55Z mwall $
# insert fields from database into template html file, resulting in a web page
# with a map and list of stations.
# Copyright 2013 Matthew Wall
#
# Run this script periodically to update the web page.

use strict;
use DBI;
use POSIX;

# location of the sqlite database
my $db = '/var/lib/weewx/stations.db';

# html template file
my $tmpl = 'stations.html.in';

# how long ago do we consider stale, in seconds
my $stale = 2_592_000; # 30 days

# format for logging
my $DATE_FORMAT = "%b %d %H:%M:%S";

while($ARGV[0]) {
    my $arg = shift;
    if ($arg eq '--template') {
        $tmpl = shift;
    } elsif ($arg eq '--stale') {
        $stale = shift;
    } elsif ($arg eq '--db') {
        $db = shift;
    }
}



# read the template file, cache in memory
my $contents = q();
if(open(IFILE, "<$tmpl")) {
    while(<IFILE>) {
        $contents .= $_;
    }
    close(IFILE);
} else {
    my $errmsg = "cannot read template file $tmpl: $!";
    errorpage($errmsg);
    logerr($errmsg);
    exit 1;
}

my @records;
my $errmsg = q();
# be sure the database is there
if (-f $db) {
    # read the database, keep only records that are not stale
    my $dbh = DBI->connect("dbi:SQLite:$db", q(), q(), { RaiseError => 0 });
    if ($dbh) {
	my $sth = $dbh->prepare('select * from stations');
	if ($sth) {
	    $sth->execute();
	    $sth->bind_columns(\my($url,$loc,$lat,$lon,$st,$ts));
	    my $now = time;
	    while($sth->fetch()) {
		next if $now - $ts > $stale;
		my %r;
		$r{url} = $url;
		$r{location} = $loc;
		$r{latitude} = $lat;
		$r{longitude} = $lon;
		$r{station_type} = $st;
		$r{last_seen} = $ts;
		push @records, \%r;
	    }
	} else {
	    $errmsg = "cannot prepare select statement: $DBI::errstr";
	    logerr($errmsg);
	}
	$dbh->disconnect();
    } else {
	$errmsg = "cannot connect to database: $DBI::errstr";
	logerr($errmsg);
    }
} else {
    $errmsg = "no database at $db";
    logerr($errmsg);
}

# inject into the template and spit it out
foreach my $line (split("\n", $contents)) {
    if($line =~ /^var sites = /) {
	if ($errmsg ne q()) {
	    print STDOUT "/* error: $errmsg */\n";
	}
        print STDOUT "var sites = [\n";
        foreach my $rec (@records) {
            print STDOUT "  { location: '$rec->{location}',\n";
            print STDOUT "    url: '$rec->{url}',\n";
            print STDOUT "    latitude: $rec->{latitude},\n";
            print STDOUT "    longitude: $rec->{longitude},\n";
            print STDOUT "    station: '$rec->{station_type}' },\n";
            print STDOUT "\n";
        }
        print STDOUT "];\n";
    } else {
        print STDOUT "$line\n";
    }
}

exit 0;


sub errorpage {
    my ($msg) = @_;
    print STDOUT "<html>\n";
    print STDOUT "<head>\n";
    print STDOUT "  <title>error</title>\n";
    print STDOUT "</head>\n";    
    print STDOUT "<body>\n";
    print STDOUT "<p>Creation of stations page failed.</p>\n";
    print STDOUT "<p>\n";
    print STDOUT "$msg\n";
    print STDOUT "</p>\n";
    print STDOUT "</body>\n";
    print STDOUT "</html>\n";
}

sub logerr {
    my ($msg) = @_;
    my $tstr = strftime $DATE_FORMAT, gmtime time;
    print STDERR "$tstr $msg\n";
}
