#!/usr/bin/perl
# $Id$
# Copyright 2013-2014 Matthew Wall, all rights reserved
#
# create thumbnail then transfer it
#
# install the Net::MQTT::Simple library

use Cwd 'abs_path';
use Time::Local;
use POSIX;
use strict;

# format for date/time in the log messages
my $DATE_FORMAT = "%Y.%m.%d %H:%M:%S";

my %cfg = get_cfg('/etc/eyesee/eyesee.cfg',
                  ('XFRIMG_TMPDIR', '/var/tmp',
                   'XFRIMG_BASEDIR', '/var/eyesee/img',
                   'XFRIMG_ID', q(),
                   'XFRIMG_LABEL', q(),
                   'XFRIMG_HOST', 'x.x.x.x',
                   'XFRIMG_DESTINATION', 'http://localhost/cgi-bin/rcvimg',
                   'XFRIMG_LAST_XFR', 'last-xfr',
                   'XFRIMG_MQTT_TELEMETRY_BROKER', 'localhost',
                   'XFRIMG_MQTT_PORT', '1883',
                   'XFRIMG_MQTT_TOPIC', 'cam/tn-upload',
                   'XFRIMG_MQTT_SEND', 0,
                  ));

my $doit = 1;            # set to 0 for debug
my $verbose = 1;
my $TMPDIR = $cfg{XFRIMG_TMPDIR};
my $basedir = $cfg{XFRIMG_BASEDIR};
my $host = $cfg{XFRIMG_HOST};
my $label = $cfg{XFRIMG_ID} ? $cfg{XFRIMG_ID} : $cfg{XFRIMG_LABEL};
my $dest = $cfg{XFRIMG_DESTINATION};
my $lastxfrfn = $cfg{XFRIMG_LAST_XFR};

while($ARGV[0]) {
    my $arg = shift;
    if ($arg eq '--dir') {
        $dir = shift;
    } elsif ($arg eq '--basedir') {
        $basedir = shift;
    } elsif ($arg eq '--host') {
        $host = shift;
    } elsif ($arg eq '--label' || $arg eq '--id') {
        $label = shift;
    } elsif ($arg eq '--dest') {
        $dest = shift;
    } elsif ($arg eq '--debug') {
        $doit = 0;
    } elsif ($arg eq '--verbose') {
        $verbose = 1;
    }
}

my $dir = q();
if ($id ne q()) {
    $dir = "$basedir/$id";
} else {
    $dir = "$basedir/$host";
}

# the filename in which to place timestamp of last upload
my $tsfn = "$dir/$lastxfrfn";

# get the image's timestamp to use as reference
my $fn = get_latest_file($dir);
if ($fn ne q()) {
    my $ts = get_ts($fn);
    my $last_ts = read_ts($tsfn);

    if ($last_ts < $ts) {
        my $src = "$TMPDIR/xfrimg_src_$$.jpg";

        # make a copy of the image for us to work on
        docmd("cp $fn $src");

        # create a shrunken version of the image
        my $tn = "$TMPDIR/xfrimg_tn_$$.jpg";
        docmd("convert $src -resize x150 $tn");

        # do the transfer
        my $labelstr = $label eq q() ? q() : "-F 'label=$label'";
        my ($fail, $rc, $sig) = docmd("curl -s -w %{http_code} $labelstr -F 'filename=$ts.jpg' -F 'thumbnail=\@$tn' -o /dev/null $dest > $src.code");
        my $code = `cat $src.code`;

        # save the timestamp after successful upload
        if ($rc == 0 && $code == 200) {
            write_ts($ts);
        }

        if ($cfg{XFRIMG_MQTT_SEND}) {
            my @t = $ts =~ m!(\d{4})(\d{2})(\d{2})(\d{2})(\d{2})(\d{2})!;
            $t[1]--;
            my $epoch = timelocal @t[5,4,3,2,1,0];
            my $topic = $cfg{XFRIMG_MQTT_TOPIC};
            my $labelstr = $label eq q() ? q() :  "\"label\":\"$label\", ";
            my $msg = "{\"ts\":$epoch, \"filename\":\"$ts.jpg\", ${labelstr}\"fail\":$fail, \"rc\":$rc, \"signal\":$sig, \"code\":$code}";
            send_mqtt($msg, $topic);
        }

        # clean up
        docmd("rm -f $src");
        docmd("rm -f $src.code");
        docmd("rm -f $tn");
    } else {
        logmsg("skipping image $fn (last:$last_ts latest:$ts)");
    }
} else {
    logmsg("cannot find latest.jpg in $dir");
}

exit 0;




# find the real path to the latest symlink
sub get_latest_file {
    my ($dir) = @_;
    my $fn = q();
    my $lfn = "$dir/latest.jpg";
    if ( -f $lfn && -l $lfn) {
        $fn = abs_path($lfn);
    }
    return $fn;
}

# get the timestamp of the latest image
sub get_ts {
    my ($fn) = @_;
    my $ts = 0;
    my ($digits) = $fn =~ /(\d+)\.jpg$/;
    if (length($digits) == 14) {
        $ts = $digits;
    }
    return $ts;
}

# save timestamp to disk.  format is single line.
sub read_ts {
    my $ts = 0;
    if (open(FILE, "<$tsfn")) {
        while(<FILE>) {
            if ($_ =~ /(\d+)/) {
                $ts = $1;
            }
        }
        close(FILE);
    } else {
        errmsg("cannot read timestamp from $tsfn: $!");
    }
    return $ts;
}

# read timestamp from disk.  format is single line.
sub write_ts {
    my ($ts) = @_;
    if (open(FILE, ">$tsfn")) {
        print FILE "$ts\n";
        close(FILE);
    } else {
        errmsg("cannot write timestamp to $tsfn: $!");
    }
}

# send an mqtt message about the transfer state
# ts is the name of the image (the time it was captured)
# fail, rc, sig is the status of the upload attempt
sub send_mqtt {
    my ($msg, $topic) = @_;

    my $rval = eval "{ require Net::MQTT::Simple; }"; ## no critic (ProhibitStringyEval)
    if (! $rval) {
        my $msg = 'Net::MQTT::Simple is not installed';
        errmsg($msg);
        return;
    }

    my $s = $cfg{XFRIMG_MQTT_TELEMETRY_BROKER} . q(:) . $cfg{XFRIMG_MQTT_PORT};
    logmsg("mqtt msg '$msg' as topic $topic at broker $s");
    my $c = Net::MQTT::Simple->new($s);
    $c->retain("$topic", $msg);
}

sub docmd() {
    my($cmd) = @_;
    logmsg($cmd);
    my $rc = -1;
    my $s = -1;
    my $fail = 0;
    if ($doit) {
        system($cmd);
        if ($? == -1) {
            $fail = 1;
            logmsg("failed to execute: $!");
        } elsif ($? & 127) {
            $s = ($? & 127);
            my $dump = ($? & 128) ? " (with coredump)" : q();
            logmsg("child died with signal $s$dump");
        } else {
            $rc = $? >> 8;
            if ($rc != 0) {
                logmsg("child exited with value $rc");
            }
        }
    }
    return ($fail, $rc, $s);
}

sub errmsg {
    my ($msg) = @_;
    my $tstr = strftime $DATE_FORMAT, localtime time;
    print "$tstr $msg\n";
}

sub logmsg {
    my ($msg) = @_;
    my $tstr = strftime $DATE_FORMAT, localtime time;
    print "$tstr $msg\n" if $verbose;
}

sub get_cfg {
    my($cfgfn, %cfg) = @_;

    for(my $i=0; $i<scalar @ARGV; $i++) {
        if ($ARGV[$i] eq '--config') {
            $i += 1;
            $cfgfn = $ARGV[$i];
        }
    }

    if (open(CFG, "<$cfgfn")) {
        while(<CFG>) {
            my $line = $_;
            next if $line =~ /^\s*\#/;
            chomp($line);
            my ($n,$v) = split('=', $line);
            $n =~ s/^\s+//g;
            $n =~ s/\s+$//g;
            $v =~ s/^\s+//g;
            $v =~ s/\s+$//g;
            $cfg{$n} = $v;
        }
        close(CFG);
    }
    return %cfg;
}
