#!/usr/bin/perl

use POSIX;
use strict;

# format for date/time in the log messages
my $DATE_FORMAT = "%Y.%m.%d %H:%M:%S";

my %cfg = get_cfg('/etc/eyesee/eyesee.cfg',
                  ('REAPER_DIR', '/var/eyesee',
                   'REAPER_MAX_AGE', 1
                  ));

my $doit = 1;
my $verbose = 1;
my $dir = $cfg{REAPER_DIR};
my $maxage = $cfg{REAPER_MAX_AGE}; # days

while($ARGV[0]) {
    my $arg = shift;
    if ($arg eq '--dir') {
        $dir = shift;
    } elsif ($arg eq '--max-age') {
        $maxage = shift;
    } elsif ($arg eq '--debug') {
        $doit = 0;
    } elsif ($arg eq '--verbose') {
        $verbose = 1;
    }
}

# figure out the latest file so we do not delete that one
my $latest = get_latest_file($dir);

# delete all of the old files
logmsg("finding files in $dir older than $maxage days");
my @files = `find $dir -type f -mtime +$maxage`;
foreach my $f (@files) {
    chomp($f);
    if ($f ne $latest) {
        docmd("rm $f");
    } else {
        logmsg("skipping latest file $f");
    }
}

# delete any remaining empty directories
logmsg("finding empty directories in $dir");
my @dirs = `find $dir -type d`;
foreach my $d (@dirs) {
    chomp($d);
    my $cnt = numfiles($d);
    if ($cnt == 0) {
        docmd("rmdir $d");
    }
}

exit 0;




sub numfiles {
    my $dirname = shift;
    my $cnt = -1;
    if (opendir(DIR, "$dirname")) {
        $cnt = scalar(grep { $_ ne "." && $_ ne ".." } readdir(DIR));
        closedir(DIR);
    } else {
        logmsg("cannot open directory $dirname: $!");
    }
    return $cnt;
}

# find the real path to the latest symlink
sub get_latest_file {
    my ($dir) = @_;
    my $fn = q();
    foreach my $lfn ("$dir/latest.jpg", "$dir/latest.jpg") {
        if ( -f $lfn && -l $lfn) {
            $fn = abs_path($lfn);
        }
    }
    return $fn;
}

sub docmd() {
    my($cmd) = @_;
    logmsg($cmd);
    my $rc = -1;
    my $s = -1;
    my $fail = 0;
    if ($doit) {
        system($cmd);
        if ($? == -1) {
            $fail = 1;
            logmsg("failed to execute: $!");
        } elsif ($? & 127) {
            $s = ($? & 127);
            my $dump = ($? & 128) ? " (with coredump)" : q();
            logmsg("child died with signal $s$dump");
        } else {
            $rc = $? >> 8;
            if ($rc != 0) {
                logmsg("child exited with value $rc");
            }
        }
    }
    return ($fail, $rc, $s);
}

sub errmsg {
    my ($msg) = @_;
    my $tstr = strftime $DATE_FORMAT, localtime time;
    print "$tstr $msg\n";
}

sub logmsg {
    my ($msg) = @_;
    my $tstr = strftime $DATE_FORMAT, localtime time;
    print "$tstr $msg\n" if $verbose;
}

sub get_cfg {
    my($cfgfn, %cfg) = @_;

    for(my $i=0; $i<scalar @ARGV; $i++) {
        if ($ARGV[$i] eq '--config') {
            $i += 1;
            $cfgfn = $ARGV[$i];
        }
    }

    if (open(CFG, "<$cfgfn")) {
        while(<CFG>) {
            my $line = $_;
            next if $line =~ /^\s*\#/;
            chomp($line);
            my ($n,$v) = split('=', $line);
            $n =~ s/^\s+//g;
            $n =~ s/\s+$//g;
            $v =~ s/^\s+//g;
            $v =~ s/\s+$//g;
            $cfg{$n} = $v;
        }
        close(CFG);
    }
    return %cfg;
}
