#!/usr/bin/perl
# $Id$
# Copyright 2013-2014 Matthew Wall, all rights reserved
#
# capture an image from a web camera
#
# one picture every 5 minutes at 30K per picture
#    288 images per day
# 105120 images per year
# 3.1GB storage for one year
#
# one picture every 2 seconds at 55kB per image
#   2.5GB per day
#    70GB per month
#
# one picture every 5 minutes at 10K to 15K per image
#      3K per day
#    100K per month
#
# take pictures only when sun is up - 45 minutes before sunrise and 45 minutes
# after sunset.
#
# in daemon mode, run continuously.
#
# images will be saved to the following directory structure:
#   /var/eyesee/img/
#   /var/eyesee/img/YYYYmmdd/YYYYmmddHHMMSS.jpg
#   /var/eyesee/img/YYYYmmdd/YYYYmmddHHMMSS.tn.jpg

use Time::Local;
use File::Basename;
use POSIX;
use strict;

# format for date/time in the log messages
my $DATE_FORMAT = "%Y.%m.%d %H:%M:%S";

my %cfg = get_cfg('/etc/eyesee/eyesee.cfg',
                  ('GETIMG_DAYONLY', 0,
                   'GETIMG_WAIT', 30,
                   'GETIMG_ID', q(),
                   'GETIMG_HOST', 'x.x.x.x',
                   'GETIMG_BASEDIR', '/var/eyesee/img',
                   'GETIMG_ODIR', q(),
                   'GETIMG_USER', 'guest',
                   'GETIMG_PASS', q(),
                   'GETIMG_DAYLIGHT_BUFFER', 45,
                   'GETIMG_LOC', q(),
                   'GETIMG_CTYPE', 'TV-IP110',
                   'GETIMG_TMPNAME', 'tmp',
                   'GETIMG_MAKE_THUMBNAIL', 1,
                   'GETIMG_THUMBNAIL_HEIGHT', 50,
                   'GETIMG_MQTT_TELEMETRY_BROKER', 'localhost',
                   'GETIMG_MQTT_PORT', '1883',
                   'GETIMG_MQTT_TOPIC', 'cam/img-capture',
                   'GETIMG_MQTT_SEND', 0,
                  ));

my $doit = 1;            # set to 0 for debug
my $verbose = 1;
my $daemon = 0;                     # if non-zero, run as a daemon
my $dayonly = $cfg{GETIMG_DAYONLY}; # take picture regardless of sunrise/sunset
my $wait = $cfg{GETIMG_WAIT};       # how often to take picture, seconds
my $host = $cfg{GETIMG_HOST};       # ip address of camera
my $id = $cfg{GETIMG_ID};           # identifier for the image source
my $basedir = $cfg{GETIMG_BASEDIR};
my $odir = $cfg{GETIMG_ODIR};
my $user = $cfg{GETIMG_USER};       # username for camera access
my $pass = $cfg{GETIMG_PASS};       # password for camera access
my $buffer = $cfg{GETIMG_DAYLIGHT_BUFFER}; # time before/after sunset, minutes
my $loc = $cfg{GETIMG_LOC};         # latitude,longitude
my $ctype = $cfg{GETIMG_CTYPE};     # default camera type
my $tmpname = $cfg{GETIMG_TMPNAME};
my $mkthumb = $cfg{GETIMG_MAKE_THUMBNAIL};
my $tnheight = $cfg{GETIMG_THUMBNAIL_HEIGHT};

while($ARGV[0]) {
    my $arg = shift;
    if ($arg eq '--id') {
        $id = shift;
    } elsif ($arg eq '--host') {
        $host = shift;
    } elsif ($arg eq '--loc') {
        $loc = shift;
    } elsif ($arg eq '--odir') {
        $odir = shift;
    } elsif ($arg eq '--user') {
        $user = shift;
    } elsif ($arg eq '--pass') {
        $pass = shift;
    } elsif ($arg eq '--cam') {
        $ctype = shift;
    } elsif ($arg eq '--ignore-daylight' || $arg eq '--force') {
        $dayonly = 0;
    } elsif ($arg eq '--daylight-only') {
        $dayonly = 1;
    } elsif ($arg eq '--daemon') {
        $daemon = 1;
    } elsif ($arg eq '--daemon-wait') {
        $wait = shift;
    } elsif ($arg eq '--debug') {
        $doit = 0;
    } elsif ($arg eq '--verbose') {
        $verbose = 1;
    }
}

my %cameras = (
    'foscam',            "http://$host/snapshot.cgi",
    'foscam FI8905W',    "http://$host/snapshot.cgi",
    'FI8905W',           "http://$host/snapshot.cgi",
    'trendnet TV-IP110', "http://$host/cgi/jpg/image.cgi",
    'TV-IP110',          "http://$host/cgi/jpg/image.cgi",
    'dlink DCS900',      "http://$host/image.jpg",
    'DCS900',            "http://$host/image.jpg",
    'dlink DCS932',      "http://$host/image/jpeg.cgi",
    'DCS932',            "http://$host/image/jpeg.cgi",
    'dahua',             "http://$host:9989/",
    'dahua-HFW1320',     "http://$host/cgi-bin/snapshot.cgi",
    'mobotix',           "http://$host/record/current.jpg",
#    'mobotix',           "http://$host/cgi-bin/image.jpg",
    'hikvision',         "http://$host/Streaming/channels/1/picture",
    );
my $img_url = q();
foreach my $c (keys %cameras) {
    if($c =~ /$ctype/) {
        $img_url = $cameras{$c};
    }
}
if($img_url eq q()) {
    errmsg("no camera found for type $ctype");
    exit 1;
}

# if no output directory specified, default to the basedir/id, or if no id
# specified default to basedir/host
if($odir eq q()) {
    if ($id ne q()) {
        $odir = "$basedir/$id";
    } else {
        $odir = "$basedir/$host";
    }
}
if(! -d $odir) {
    logmsg("creating directory $odir");
    docmd("mkdir -p $odir");
}

do {
    my $now = time;
    my $skip = 0;

    if ($dayonly) {
        my $fail = 0;
        eval { require DateTime; };
        if ($@) {
            errmsg("DateTime is not installed");
            $skip = 1;
            $fail = 1;
        }
        eval { require DateTime::Event::Sunrise; };
        if ($@) {
            errmsg("DateTime::Event::Sunrise is not installed");
            $skip = 1;
            $fail = 1;
        }
        if($loc eq q()) {
            errmsg("no lat/lon specified");
            $skip = 1;
            $fail = 1;
        }
        if (! $fail) {
            my($lat,$lon) = split(',', $loc);
            my $dt = DateTime->now();
            $dt->set_time_zone('local');
            my $s = DateTime::Event::Sunrise->new(longitude => $lon,
                                                  latitude => $lat);
            my $span = $s->sunrise_sunset_span($dt);
            my($y,$m,$d,$H,$M,$S) =
                $span->start->datetime =~ /(\d\d\d\d)-(\d\d)-(\d\d)T(\d\d):(\d\d):(\d\d)/;
            my $sr = DateTime->new(
                year => $y, month => $m, day => $d,
                hour => $H, minute => $M, second => $S,
                time_zone => 'local'
                );
            $sr->add(minutes => -$buffer);
            ($y,$m,$d,$H,$M,$S) =
                $span->end->datetime =~ /(\d\d\d\d)-(\d\d)-(\d\d)T(\d\d):(\d\d):(\d\d)/;
            my $ss = DateTime->new(
                year => $y, month => $m, day => $d,
                hour => $H, minute => $M, second => $S,
                time_zone => 'local'
                );
            $ss->add(minutes => $buffer);
            $span = DateTime::Span->from_datetimes(start => $sr, end => $ss);
            if (! $span->contains($dt)) {
                errmsg("skipping due to darkness (now=$dt sr=$sr ss=$ss)");
                $skip = 1;
            }
        }
    }

    if (! $skip) {
        my ($fail, $rc, $sig) = docmd("curl -s -S -u $user:$pass -w %{http_code} -o $odir/$tmpname ${img_url} > $odir/$tmpname.code");
        my $code = `cat $odir/$tmpname.code`;
        # final filename is the timestamp
        my $ts = strftime("%Y%m%d%H%M%S", localtime($now));
        # be sure that curl executed properly and we got the right http return
        if ($fail == 0 && $rc == 0 && "$code" eq "200") {

            # keep images in one directory per day YYYYmmdd
            my($subdir) = $ts =~ /(\d\d\d\d\d\d\d\d)/;
            my $ofile = "$ts.jpg";

            # create the directory if it does not already exist
            if(! -d "$odir/$subdir") {
                logmsg("creating directory $odir/$subdir");
                docmd("mkdir -p $odir/$subdir");
            }

            docmd("mv $odir/$tmpname $odir/$subdir/$ofile");
            docmd("rm $odir/$tmpname.code");
            docmd("rm -f $odir/latest.jpg");
            docmd("ln -s $subdir/$ofile $odir/latest.jpg");

            # create a shrunken version of the image
            if ($mkthumb && -f "$odir/$subdir/$ofile") {
                my $tn = "$odir/$subdir/${ts}-tn.jpg";
                docmd("convert $odir/$subdir/$ofile -resize x${tnheight} $tn");
            }
        } else {
            errmsg("download failed: fail=$fail rc=$rc code=$code");
        }

        if ($cfg{GETIMG_MQTT_SEND}) {
            my $msg = "{\"ts\":$now, \"filename\":\"$ts.jpg\", \"fail\":$fail, \"rc\":$rc, \"signal\":$sig, \"code\":\"$code\"}";
            send_mqtt($msg);
        }
    }

    if ($daemon) {
        sleep($wait);
    }
} while($daemon);

exit 0;




# send an mqtt message about the transfer state
# ts is the name of the image (the time it was captured)
# fail, rc, sig is the status of the upload attempt
sub send_mqtt {
    my ($msg) = @_;

    my $rval = eval "{ require Net::MQTT::Simple; }"; ## no critic (ProhibitStringyEval)
    if (! $rval) {
        my $msg = 'Net::MQTT::Simple is not installed';
        errmsg($msg);
        return;
    }

    my $topic = $cfg{GETIMG_MQTT_TOPIC};
    my $s = $cfg{GETIMG_MQTT_TELEMETRY_BROKER} . q(:) . $cfg{GETIMG_MQTT_PORT};
    logmsg("mqtt msg '$msg' as topic $topic at broker $s");
    my $c = Net::MQTT::Simple->new($s);
    $c->retain("$topic", $msg);
}




sub docmd() {
    my($cmd) = @_;
    logmsg($cmd);
    my $rc = -1;
    my $s = -1;
    my $fail = 0;
    if ($doit) {
        system($cmd);
        if ($? == -1) {
            $fail = 1;
            logmsg("failed to execute: $!");
        } elsif ($? & 127) {
            $s = ($? & 127);
            my $dump = ($? & 128) ? " (with coredump)" : q();
            logmsg("child died with signal $s$dump");
        } else {
            $rc = $? >> 8;
            if ($rc != 0) {
                logmsg("child exited with value $rc");
            }
        }
    }
    return ($fail, $rc, $s);
}

sub errmsg {
    my ($msg) = @_;
    my $tstr = strftime $DATE_FORMAT, localtime time;
    print "$tstr $msg\n";
}

sub logmsg {
    my ($msg) = @_;
    my $tstr = strftime $DATE_FORMAT, localtime time;
    print "$tstr $msg\n" if $verbose;
}

sub get_cfg {
    my($cfgfn, %cfg) = @_;

    for(my $i=0; $i<scalar @ARGV; $i++) {
        if ($ARGV[$i] eq '--config') {
            $i += 1;
            $cfgfn = $ARGV[$i];
        }
    }

    if (open(CFG, "<$cfgfn")) {
        while(<CFG>) {
            my $line = $_;
            next if $line =~ /^\s*\#/;
            chomp($line);
            my ($n,$v) = split('=', $line);
            $n =~ s/^\s+//g;
            $n =~ s/\s+$//g;
            $v =~ s/^\s+//g;
            $v =~ s/\s+$//g;
            $cfg{$n} = $v;
        }
        close(CFG);
    }
    return %cfg;
}
