#!/usr/bin/perl

# License: OSI Artistic License
#          http://www.opensource.org/licenses/artistic-license.php
# Author:  (c) 2004 Soren Dossing
# Author:  (c) 2005 Robert Teeter
# Author:  (c) 2008 Alan Brenner, Ithaka Harbors
# Author:  (c) 2010 Matthew Wall

# The configuration file and ngshared.pm must be in this directory.
# So take note upgraders, there is no $configfile = '....' line anymore.
use lib '/opt/nagiosgraph/etc';

# Main program - change nothing below

use ngshared qw(:SHOWHOST);

use CGI;
use English qw(-no_match_vars);
use File::Find;
use strict;
use warnings;

readconfig('read');
if (defined $Config{ngshared}) {
    debug(1, $Config{ngshared});
    htmlerror($Config{ngshared});
    exit;
}

my $cgi = new CGI;  ## no critic (ProhibitIndirectSyntax)
$cgi->autoEscape(0);

my $params = getparams($cgi);
getdebug('showhost', $params->{host}, $params->{service});

dumper(DBDEB, 'config', \%Config);
dumper(DBDEB, 'params', $params);

my $host = q();
if ($params->{host}) { $host = $params->{host}; }

my $periods = getperiods('timehost', $params->{period});

my @style;
if ($Config{stylesheet}) {
    @style = (-style => {-src => "$Config{stylesheet}"});
}

find(\&getgraphlist, $Config{rrddir});

my $hdb = readhostdb($host);

# nagios and nagiosgraph may not share the same cgi directory
my $nagioscgiurl = $Config{nagiosgraphcgiurl};
if ($Config{nagioscgiurl}) { $nagioscgiurl = $Config{nagioscgiurl}; }

# draw the page
print printheader($cgi,
                  { title => $host,
                    style => \@style,
                    call => 'host',
                    default => $host,
                    label => $cgi->a({href => $nagioscgiurl . '/extinfo.cgi?type=1&host=' . $host}, $host)}) or
    debug(DBCRT, "error sending HTML to web server: $OS_ERROR");

my $now = time;
foreach my $period (graphsizes($periods)) {
    dumper(DBDEB, 'period', $period);
    my $str = q();
    foreach my $dbinfo (@{$hdb}) {
        $dbinfo->{host} = $params->{host};
        cfgparams($dbinfo, $params, $dbinfo->{service});
        dumper(DBDEB, 'dbinfo', $dbinfo);

        my $url = $Config{nagiosgraphcgiurl} .
            '/showservice.cgi?service=' . $dbinfo->{service} .
            join('&db=' . @{$dbinfo->{db}});
        $url =~ tr/ /+/;

        my $link = $cgi->a({href => $url},
                           defined $dbinfo->{label} ? $dbinfo->{label} :
                           $dbinfo->{service});
        $str .= printgraphlinks($cgi, $dbinfo, $period, $link) . "\n";
    }
    print printperiodlinks($cgi, $params, $period, $now, $str) or
        debug(DBCRT, "error sending HTML to web server: $OS_ERROR");
}

print printscript('host', $host, $params->{service});

print printfooter($cgi) or
    debug(DBCRT, "error sending HTML to web server: $OS_ERROR");

__END__

=head1 NAME

showhost.cgi - Graph Nagios data for a given host

=head1 DESCRIPTION

This is a CGI script that is designed to be run on a web server.  It generates
a page of HTML that displays a list of graphs for a single host.  The graph
data are retrieved from RRD files and are typically captured by insert.pl.

The showgraph.cgi script generates the graphs themselves.

=head1 USAGE

B<showhost.cgi>?host=host_name

=head1 CONFIGURATION

The B<nagiosgraph.conf> file controls the behavior of this script.

The B<hostdb.conf> controls which services will be listed and the order in
which those services will appear.

=head1 OPTIONS

host=host_name

period=(day week month quarter year)

=head1 DIAGNOSTICS

Use the debug_showhost setting from B<nagiosgraph.conf> to control the amount
of debug information that will be emitted by this script.  Debug output will
go to the web server error log.

=head1 DEPENDENCIES

=over 4

=item B<showgraph.cgi>

This generates the graphs shown in the HTML generated here.

=item B<Nagios>

While this could probably run without Nagios, as long as RRD databases exist,
it is intended to work along side Nagios.

=item B<rrdtool>

This provides the data storage and graphing system.

=item B<RRDs>

This provides the perl interface to rrdtool.

=back

=head1 INSTALLATION

Copy this file into Nagios' cgi directory (for the Apache web server, see where
the ScriptAlias for /nagios/cgi-bin points), and make sure it is executable by
the web server.

Install the B<ngshared.pm> file and edit this file to change the B<use lib>
line to point to the directory containing B<ngshared.pm>.

Create or edit the example B<hostdb.conf>, which must reside in the same
directory as B<ngshared.pm>.

To link a web page generated by this script from Nagios, add

=over 4

action_url https://server/nagios/cgi-bin/showhost.cgi?host=host1

=back

to the B<define host> (Nagios 3) or B<define hostextinfo> (Nagios 2.12) stanza
(changing the base URL and host1 as needed).

Copy the images/action.gif file to the nagios/images directory, if desired.

=head1 SEE ALSO

B<hostdb.conf> B<nagiosgraph.conf> B<showgraph.cgi> B<showservice.cgi> B<ngshared.pm>

=head1 AUTHOR

Soren Dossing, the original author of show.cgi in 2004.

Robert Teeter, the original author of showhost.cgi in 2005

Alan Brenner - alan.brenner@ithaka.org; I've updated this from the version
at http://nagiosgraph.wiki.sourceforge.net/ by moving some subroutines into a
shared file (ngshared.pm), using showgraph.cgi, and adding links for show.cgi
and showservice.cgi.

Matthew Wall, added features, bug fixes and refactoring in 2010.

=head1 LICENSE AND COPYRIGHT

Copyright (C) 2005 Soren Dossing, 2008 Ithaka Harbors, Inc.

This program is free software; you can redistribute it and/or modify it
under the terms of the OSI Artistic License:
http://www.opensource.org/licenses/artistic-license-2.0.php

This program is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE.
